/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.tests.common.spi;

import java.util.Arrays;
import java.util.Properties;

import junit.framework.TestCase;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.SPISession;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.TranslatableString;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Group;
import org.w3c.dom.Element;

public class Group_Test extends TestCase {
    Group group;
    Group clonedGroup;
    Element element;
    SPISessionImpl session;

    public Group_Test(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
        
        group = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createGroup("groupID", Arrays.asList("feature1", "feature2"));
        
        group.setDefault(Boolean.TRUE);
        group.setDisplayName(new TranslatableString("displayName", "displayNameKey"));
        group.setDescription(new TranslatableString("desc", "descKey"));
        group.setShortDescription(new TranslatableString("shortDesc", "shortDescKey"));
        
        clonedGroup = (Group)group.clone();
        element = ((SPIDataObject)group).getElement();
        session = (SPISessionImpl)SPISession.DEFAULT_INSTANCE;
    }

    public void testGetTranslatableStrings() {
        Properties props = group.getTranslatableStrings();
        Properties clonedProps = clonedGroup.getTranslatableStrings();
        
        assertEquals(3, props.keySet().size());
        assertEquals(3, clonedProps.keySet().size());
        
        assertEquals("desc", props.get("descKey"));
        assertEquals("shortDesc", props.get("shortDescKey"));
        assertEquals("displayName", props.get("displayNameKey"));
        
        assertEquals("desc", clonedProps.get("descKey"));
        assertEquals("shortDesc", clonedProps.get("shortDescKey"));
        assertEquals("displayName", clonedProps.get("displayNameKey"));
    }

    public void testGetDefault() {
        assertTrue(group.getDefault().booleanValue());
        assertTrue(clonedGroup.getDefault().booleanValue());
        assertTrue(Boolean.parseBoolean(element.getAttribute("default")));
    }

    public void testGetGroupID() {
        assertEquals("groupID", group.getGroupID());
        assertEquals("groupID", clonedGroup.getGroupID());
        assertEquals("groupID", element.getAttribute("id"));
    }

    public void testGetSelectedFeatures() {
        assertEquals(2, group.getSelectedFeatures().size());
        assertEquals("feature1", group.getSelectedFeatures().toArray(new String[0])[0]);
        assertEquals("feature2", group.getSelectedFeatures().toArray(new String[0])[1]);

        assertEquals(2, clonedGroup.getSelectedFeatures().size());
        assertEquals("feature1", clonedGroup.getSelectedFeatures().toArray(new String[0])[0]);
        assertEquals("feature2", clonedGroup.getSelectedFeatures().toArray(new String[0])[1]);
        
        assertEquals(2, session.getChildren(element, "SelectedFeature").size());
        assertEquals("feature1", ( session.getChildren(element, "SelectedFeature").toArray(new Element[0])[0]).getAttribute("featureRef"));
        assertEquals("feature2", ( session.getChildren(element, "SelectedFeature").toArray(new Element[0])[1]).getAttribute("featureRef"));
    }

    public void testGetDisplayName() {
        assertEquals("displayName", group.getDisplayName().getValue());
        assertEquals("displayNameKey", group.getDisplayName().getTranslationKey());

        assertEquals("displayName", clonedGroup.getDisplayName().getValue());
        assertEquals("displayNameKey", clonedGroup.getDisplayName().getTranslationKey());
        
        assertEquals("displayName", ( session.getChildren(element, "DisplayName").toArray(new Element[0])[0]).getTextContent());
        assertEquals("displayNameKey", ( session.getChildren(element, "DisplayName").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testGetDescription() {
        assertEquals("desc", group.getDescription().getValue());
        assertEquals("descKey", group.getDescription().getTranslationKey());

        assertEquals("desc", clonedGroup.getDescription().getValue());
        assertEquals("descKey", clonedGroup.getDescription().getTranslationKey());
        
        assertEquals("desc", ( session.getChildren(element, "Description").toArray(new Element[0])[0]).getTextContent());
        assertEquals("descKey", ( session.getChildren(element, "Description").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testGetShortDescription() {
        assertEquals("shortDesc", group.getShortDescription().getValue());
        assertEquals("shortDescKey", group.getShortDescription().getTranslationKey());

        assertEquals("shortDesc", clonedGroup.getShortDescription().getValue());
        assertEquals("shortDescKey", clonedGroup.getShortDescription().getTranslationKey());
        
        assertEquals("shortDesc", ( session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getTextContent());
        assertEquals("shortDescKey", ( session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testSetDefault() {
        group.setDefault(Boolean.FALSE);
        
        assertFalse(group.getDefault().booleanValue());
        assertFalse(Boolean.parseBoolean(element.getAttribute("default")));
    }

    public void testSetGroupID() {
        group.setGroupID("differentID");
        
        assertEquals("differentID", group.getGroupID());
        assertEquals("differentID", element.getAttribute("id"));
        
        try {
            group.setGroupID(null);
            fail("ID was set to null");
        }
        catch (Exception e) {
        }
    }

    public void testSetSelectedFeatures() {
        group.setSelectedFeatures(Arrays.asList("feature3"));
        
        assertEquals(1, group.getSelectedFeatures().size());
        assertEquals("feature3", group.getSelectedFeatures().toArray(new String[0])[0]);

        assertEquals(1, session.getChildren(element, "SelectedFeature").size());
        assertEquals("feature3", ( session.getChildren(element, "SelectedFeature").toArray(new Element[0])[0]).getAttribute("featureRef"));
        
        try {
            group.setSelectedFeatures(null);
            fail("SelectedFeatures set to null");
        }
        catch (Exception e) {
        }
    }

    public void testSetDisplayName() {
        group.setDisplayName(new TranslatableString("differentName", "differentNameKey"));
        
        assertEquals("differentName", group.getDisplayName().getValue());
        assertEquals("differentNameKey", group.getDisplayName().getTranslationKey());

        assertEquals("differentName", ( session.getChildren(element, "DisplayName").toArray(new Element[0])[0]).getTextContent());
        assertEquals("differentNameKey", ( session.getChildren(element, "DisplayName").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testSetDescription() {
        group.setDescription(new TranslatableString("differentDesc", "differentDescKey"));
        
        assertEquals("differentDesc", group.getDescription().getValue());
        assertEquals("differentDescKey", group.getDescription().getTranslationKey());

        assertEquals("differentDesc", ( session.getChildren(element, "Description").toArray(new Element[0])[0]).getTextContent());
        assertEquals("differentDescKey", ( session.getChildren(element, "Description").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testSetShortDescription() {
        group.setShortDescription(new TranslatableString("differentShortDesc", "differentShortDescKey"));

        assertEquals("differentShortDesc", group.getShortDescription().getValue());
        assertEquals("differentShortDescKey", group.getShortDescription().getTranslationKey());

        assertEquals("differentShortDesc", ( session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getTextContent());
        assertEquals("differentShortDescKey", ( session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }
}
