/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.tests.common.spi;

import java.math.BigInteger;
import java.util.Arrays;
import java.util.Collection;
import java.util.Properties;

import junit.framework.TestCase;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.artifact.ArtifactArgument;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ReferencedPackage;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ResourceMap;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ResultingChangeMap;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ResultingResourceMap;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.SPISession;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ContentSelectionFeature;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Language;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.Requirement;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.OutputVariable;
import org.w3c.dom.Element;

public class ReferencedPackage_Test extends TestCase {
    ReferencedPackage referencedPackage;
    ReferencedPackage clonedReferencedPackage;
    Element element;
    SPISessionImpl session;

    public ReferencedPackage_Test(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
        
        referencedPackage = SPISession.DEFAULT_INSTANCE.createBaseFactory().createReferencedPackage("referencedPackageContentRef", "referencedPackageID");

        ArtifactArgument argument1 = new ArtifactArgument("arg1Name", "arg1Value", new Boolean(false));
        ArtifactArgument argument2 = new ArtifactArgument("arg2Name", "arg2Value", new Boolean(true));
        referencedPackage.setArguments(Arrays.asList(argument1, argument2));
        
        Language language1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("lang1Type");
        Language language2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("lang2Type");
        referencedPackage.setLanguages(Arrays.asList(language1, language2));
        
        referencedPackage.setOperation("argumentOperation");
        
        OutputVariable var1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createOutputVariable("var1ID");
        OutputVariable var2 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createOutputVariable("var2ID");
        referencedPackage.setOutputVariables(Arrays.asList(var1, var2));

        ContentSelectionFeature feat1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createContentSelectionFeature("feature1");
        ContentSelectionFeature feat2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createContentSelectionFeature("feature2");
        referencedPackage.setRequiredFeatures(Arrays.asList(feat1, feat2));

        referencedPackage.setRequiredGroup("groupName");
        
        ResourceMap map1 = SPISession.DEFAULT_INSTANCE.createBaseFactory().createRequiredResourceMap("map1Ref");
        ResourceMap map2 = SPISession.DEFAULT_INSTANCE.createBaseFactory().createRequiredResourceMap("map2Ref");
        referencedPackage.setRequiredResourceMaps(Arrays.asList(map1, map2));

        Requirement req1 = SPISession.DEFAULT_INSTANCE.createRequirementFactory().createRequirement("req1ID");
        Requirement req2 = SPISession.DEFAULT_INSTANCE.createRequirementFactory().createRequirement("req2ID");
        referencedPackage.setRequirements(Arrays.asList(req1, req2));
        
        ResultingChangeMap change1 = SPISession.DEFAULT_INSTANCE.createBaseFactory().createResultingChangeMap("change1Ref");
        ResultingChangeMap change2 = SPISession.DEFAULT_INSTANCE.createBaseFactory().createResultingChangeMap("change2Ref");
        referencedPackage.setResultingChangeMaps(Arrays.asList(change1, change2));

        ResultingResourceMap res1 = SPISession.DEFAULT_INSTANCE.createBaseFactory().createResultingResourceMap("res1Ref");
        ResultingResourceMap res2 = SPISession.DEFAULT_INSTANCE.createBaseFactory().createResultingResourceMap("res2Ref");
        referencedPackage.setResultingResourceMaps(Arrays.asList(res1, res2));

        referencedPackage.setWeight(new BigInteger("2"));
        
        clonedReferencedPackage = (ReferencedPackage)referencedPackage.clone();
        element = ((SPIDataObject)referencedPackage).getElement();
        session = (SPISessionImpl)SPISession.DEFAULT_INSTANCE;
    }

    public void testGetArguments() {
        Collection<ArtifactArgument> arguments = referencedPackage.getArguments();
        Collection<ArtifactArgument> clonedArguments = clonedReferencedPackage.getArguments();
        Collection<Element> dataObjectArguments = session.getChildren(session.getChild(element, "Arguments"), "Argument");
        
        assertEquals(2, arguments.size());
        assertEquals(2, clonedArguments.size());
        assertEquals(2, dataObjectArguments.size());
        
        assertEquals("arg1Name", arguments.toArray(new ArtifactArgument[0])[0].getName());
        assertEquals("arg1Value", arguments.toArray(new ArtifactArgument[0])[0].getValue());
        assertFalse(arguments.toArray(new ArtifactArgument[0])[0].getRequired().booleanValue());
        assertEquals("arg2Name", arguments.toArray(new ArtifactArgument[0])[1].getName());
        assertEquals("arg2Value", arguments.toArray(new ArtifactArgument[0])[1].getValue());
        assertTrue(arguments.toArray(new ArtifactArgument[0])[1].getRequired().booleanValue());

        assertEquals("arg1Name", clonedArguments.toArray(new ArtifactArgument[0])[0].getName());
        assertEquals("arg1Value", clonedArguments.toArray(new ArtifactArgument[0])[0].getValue());
        assertFalse(clonedArguments.toArray(new ArtifactArgument[0])[0].getRequired().booleanValue());
        assertEquals("arg2Name", clonedArguments.toArray(new ArtifactArgument[0])[1].getName());
        assertEquals("arg2Value", clonedArguments.toArray(new ArtifactArgument[0])[1].getValue());
        assertTrue(clonedArguments.toArray(new ArtifactArgument[0])[1].getRequired().booleanValue());
        
        assertEquals("arg1Name", (dataObjectArguments.toArray(new Element[0])[0]).getAttribute("name"));
        assertEquals("arg1Value", (dataObjectArguments.toArray(new Element[0])[0]).getAttribute("value"));
        assertFalse(Boolean.parseBoolean((dataObjectArguments.toArray(new Element[0])[0]).getAttribute("required")));
        assertEquals("arg2Name", (dataObjectArguments.toArray(new Element[0])[1]).getAttribute("name"));
        assertEquals("arg2Value", (dataObjectArguments.toArray(new Element[0])[1]).getAttribute("value"));
        assertTrue(Boolean.parseBoolean((dataObjectArguments.toArray(new Element[0])[1]).getAttribute("required")));
    }

    public void testGetReferencedPackageID() {
        assertEquals("referencedPackageID", referencedPackage.getReferencedPackageID());
        assertEquals("referencedPackageID", clonedReferencedPackage.getReferencedPackageID());
        assertEquals("referencedPackageID", element.getAttribute("id"));
    }

    public void testGetOperation() {
        assertEquals("argumentOperation", referencedPackage.getOperation());
        assertEquals("argumentOperation", clonedReferencedPackage.getOperation());
        assertEquals("argumentOperation", element.getAttribute("operation"));
    }

    public void testGetContentReference() {
        assertEquals("referencedPackageContentRef", referencedPackage.getContentReference());
        assertEquals("referencedPackageContentRef", clonedReferencedPackage.getContentReference());
        assertEquals("referencedPackageContentRef", element.getAttribute("contentRef"));
    }

    public void testGetLanguages() {
        Collection<Language> languages = referencedPackage.getLanguages();
        Collection<Language> clonedLanguages = clonedReferencedPackage.getLanguages();
        Collection<Element> dataObjectLanguages = session.getChildren(session.getChild(element, "Languages"), "Language");
        
        assertEquals(2, languages.size());
        assertEquals(2, clonedLanguages.size());
        assertEquals(2, dataObjectLanguages.size());

        assertEquals("lang1Type", languages.toArray(new Language[0])[0].getType());
        assertEquals("lang2Type", languages.toArray(new Language[0])[1].getType());

        assertEquals("lang1Type", clonedLanguages.toArray(new Language[0])[0].getType());
        assertEquals("lang2Type", clonedLanguages.toArray(new Language[0])[1].getType());
        
        assertEquals("lang1Type", (dataObjectLanguages.toArray(new Element[0])[0]).getAttribute("type"));
        assertEquals("lang2Type", (dataObjectLanguages.toArray(new Element[0])[1]).getAttribute("type"));
    }

    public void testGetOutputVariables() {
        Collection<OutputVariable> variables = referencedPackage.getOutputVariables();
        Collection<OutputVariable> clonedVariables = clonedReferencedPackage.getOutputVariables();
        Collection<Element> dataObjectVariables = session.getChildren(session.getChild(element, "OutputVariables"), "OutputVariable");
        
        assertEquals(2, variables.size());
        assertEquals(2, clonedVariables.size());
        assertEquals(2, dataObjectVariables.size());

        assertEquals("var1ID", variables.toArray(new OutputVariable[0])[0].getVariableID());
        assertEquals("var2ID", variables.toArray(new OutputVariable[0])[1].getVariableID());

        assertEquals("var1ID", clonedVariables.toArray(new OutputVariable[0])[0].getVariableID());
        assertEquals("var2ID", clonedVariables.toArray(new OutputVariable[0])[1].getVariableID());
        
        assertEquals("var1ID", (dataObjectVariables.toArray(new Element[0])[0]).getAttribute("id"));
        assertEquals("var2ID", (dataObjectVariables.toArray(new Element[0])[1]).getAttribute("id"));
    }

    public void testGetResultingChangeMaps() {
        Collection<ResultingChangeMap> maps = referencedPackage.getResultingChangeMaps();
        Collection<ResultingChangeMap> clonedMaps = clonedReferencedPackage.getResultingChangeMaps();
        Collection<Element> dataObjectMaps = session.getChildren(element, "ResultingChangeMap");
        
        assertEquals(2, maps.size());
        assertEquals(2, clonedMaps.size());
        assertEquals(2, dataObjectMaps.size());

        assertEquals("change1Ref", maps.toArray(new ResultingChangeMap[0])[0].getResourceReference());
        assertEquals("change2Ref", maps.toArray(new ResultingChangeMap[0])[1].getResourceReference());

        assertEquals("change1Ref", clonedMaps.toArray(new ResultingChangeMap[0])[0].getResourceReference());
        assertEquals("change2Ref", clonedMaps.toArray(new ResultingChangeMap[0])[1].getResourceReference());
        
        assertEquals("change1Ref", (dataObjectMaps.toArray(new Element[0])[0]).getAttribute("resourceRef"));
        assertEquals("change2Ref", (dataObjectMaps.toArray(new Element[0])[1]).getAttribute("resourceRef"));
    }

    public void testGetResultingResourceMaps() {
        Collection<ResultingResourceMap> maps = referencedPackage.getResultingResourceMaps();
        Collection<ResultingResourceMap> clonedMaps = clonedReferencedPackage.getResultingResourceMaps();
        Collection<Element> dataObjectMaps = session.getChildren(element, "ResultingResourceMap");
        
        assertEquals(2, maps.size());
        assertEquals(2, clonedMaps.size());
        assertEquals(2, dataObjectMaps.size());

        assertEquals("res1Ref", maps.toArray(new ResultingResourceMap[0])[0].getResourceReference());
        assertEquals("res2Ref", maps.toArray(new ResultingResourceMap[0])[1].getResourceReference());

        assertEquals("res1Ref", clonedMaps.toArray(new ResultingResourceMap[0])[0].getResourceReference());
        assertEquals("res2Ref", clonedMaps.toArray(new ResultingResourceMap[0])[1].getResourceReference());
        
        assertEquals("res1Ref", (dataObjectMaps.toArray(new Element[0])[0]).getAttribute("resourceRef"));
        assertEquals("res2Ref", (dataObjectMaps.toArray(new Element[0])[1]).getAttribute("resourceRef"));
    }

    public void testGetRequiredResourceMaps() {
        Collection<ResourceMap> maps = referencedPackage.getRequiredResourceMaps();
        Collection<ResourceMap> clonedMaps = clonedReferencedPackage.getRequiredResourceMaps();
        Collection<Element> dataObjectMaps = session.getChildren(element, "RequiredResourceMap");
        
        assertEquals(2, maps.size());
        assertEquals(2, clonedMaps.size());
        assertEquals(2, dataObjectMaps.size());

        assertEquals("map1Ref", maps.toArray(new ResourceMap[0])[0].getResourceReference());
        assertEquals("map2Ref", maps.toArray(new ResourceMap[0])[1].getResourceReference());

        assertEquals("map1Ref", clonedMaps.toArray(new ResourceMap[0])[0].getResourceReference());
        assertEquals("map2Ref", clonedMaps.toArray(new ResourceMap[0])[1].getResourceReference());
        
        assertEquals("map1Ref", (dataObjectMaps.toArray(new Element[0])[0]).getAttribute("resourceRef"));
        assertEquals("map2Ref", (dataObjectMaps.toArray(new Element[0])[1]).getAttribute("resourceRef"));
    }

    public void testGetRequiredFeatures() {
        Collection<ContentSelectionFeature> features = referencedPackage.getRequiredFeatures();
        Collection<ContentSelectionFeature> clonedFeatures = clonedReferencedPackage.getRequiredFeatures();
        Collection<Element> dataObjectFeatures = session.getChildren(session.getChild(element, "RequiredContentSelection"), "Feature");
        
        assertEquals(2, features.size());
        assertEquals(2, clonedFeatures.size());
        assertEquals(2, dataObjectFeatures.size());

        assertEquals("feature1", features.toArray(new ContentSelectionFeature[0])[0].getFeature());
        assertEquals("feature2", features.toArray(new ContentSelectionFeature[0])[1].getFeature());

        assertEquals("feature1", clonedFeatures.toArray(new ContentSelectionFeature[0])[0].getFeature());
        assertEquals("feature2", clonedFeatures.toArray(new ContentSelectionFeature[0])[1].getFeature());
        
        assertEquals("feature1", (dataObjectFeatures.toArray(new Element[0])[0]).getTextContent());
        assertEquals("feature2", (dataObjectFeatures.toArray(new Element[0])[1]).getTextContent());
    }

    public void testGetRequiredGroup() {
        assertEquals("groupName", referencedPackage.getRequiredGroup());
        assertEquals("groupName", clonedReferencedPackage.getRequiredGroup());
        assertEquals("groupName", session.getChild(session.getChild(((SPIDataObject)referencedPackage).getElement(), "RequiredContentSelection"), "Group").getTextContent());
    }

    public void testGetRequirements() {
        Collection<Requirement> reqs = referencedPackage.getRequirements();
        Collection<Requirement> clonedReqs = clonedReferencedPackage.getRequirements();
        Collection<Element> dataObjectReqs = session.getChildren(session.getChild(element, "Requirements"), "Requirement");
        
        assertEquals(2, reqs.size());
        assertEquals(2, clonedReqs.size());
        assertEquals(2, dataObjectReqs.size());

        assertEquals("req1ID", reqs.toArray(new Requirement[0])[0].getRequirementID());
        assertEquals("req2ID", reqs.toArray(new Requirement[0])[1].getRequirementID());

        assertEquals("req1ID", clonedReqs.toArray(new Requirement[0])[0].getRequirementID());
        assertEquals("req2ID", clonedReqs.toArray(new Requirement[0])[1].getRequirementID());
        
        assertEquals("req1ID", (dataObjectReqs.toArray(new Element[0])[0]).getAttribute("id"));
        assertEquals("req2ID", (dataObjectReqs.toArray(new Element[0])[1]).getAttribute("id"));
    }

    public void testGetWeight() {
        assertEquals(2, referencedPackage.getWeight().intValue());
        assertEquals(2, clonedReferencedPackage.getWeight().intValue());
        assertEquals(2, new BigInteger(element.getAttribute("weight")).intValue());
    }

    public void testSetArguments() {
        ArtifactArgument arg = new ArtifactArgument("differentArgName", "differentArgValue", new Boolean(true));
        referencedPackage.setArguments(Arrays.asList(arg));

        Collection<ArtifactArgument> arguments = referencedPackage.getArguments();
        Collection<Element> dataObjectArguments = session.getChildren(session.getChild(element, "Arguments"), "Argument");
        	
        assertEquals(1, arguments.size());
        assertEquals(1, dataObjectArguments.size());
        
        assertEquals("differentArgName", arguments.toArray(new ArtifactArgument[0])[0].getName());
        assertEquals("differentArgValue", arguments.toArray(new ArtifactArgument[0])[0].getValue());
        assertTrue(arguments.toArray(new ArtifactArgument[0])[0].getRequired().booleanValue());

        assertEquals("differentArgName", (dataObjectArguments.toArray(new Element[0])[0]).getAttribute("name"));
        assertEquals("differentArgValue", (dataObjectArguments.toArray(new Element[0])[0]).getAttribute("value"));
        assertTrue(Boolean.parseBoolean((dataObjectArguments.toArray(new Element[0])[0]).getAttribute("required")));
    }

    public void testSetReferencedPackageID() {
        referencedPackage.setReferencedPackageID("differentID");
        
        assertEquals("differentID", referencedPackage.getReferencedPackageID());
        assertEquals("differentID", element.getAttribute("id"));
        
        try {
            referencedPackage.setReferencedPackageID(null);
            fail("Null value was set for id");
        }
        catch(Exception e) {
        }
    }

    public void testSetOperation() {
        referencedPackage.setOperation("differentOperation");

        assertEquals("differentOperation", referencedPackage.getOperation());
        assertEquals("differentOperation", element.getAttribute("operation"));
    }

    public void testSetContentReference() {
        referencedPackage.setContentReference("differentRef");

        assertEquals("differentRef", referencedPackage.getContentReference());
        assertEquals("differentRef", element.getAttribute("contentRef"));

        try {
            referencedPackage.setReferencedPackageID(null);
            fail("Null value was set for contentRef");
        }
        catch(Exception e) {
        }
    }

    public void testSetLanguages() {
        Language lang = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("differentType");
        referencedPackage.setLanguages(Arrays.asList(lang));

        Collection<Language> languages = referencedPackage.getLanguages();
        Collection<Element> dataObjectLanguages = session.getChildren(session.getChild(element, "Languages"), "Language");
        
        assertEquals(1, languages.size());
        assertEquals(1, dataObjectLanguages.size());

        assertEquals("differentType", languages.toArray(new Language[0])[0].getType());

        assertEquals("differentType", (dataObjectLanguages.toArray(new Element[0])[0]).getAttribute("type"));
    }

    public void testSetOutputVariables() {
        OutputVariable var = SPISession.DEFAULT_INSTANCE.createVariableFactory().createOutputVariable("differentID");
        referencedPackage.setOutputVariables(Arrays.asList(var));
        
        Collection<OutputVariable> variables = referencedPackage.getOutputVariables();
        Collection<Element> dataObjectVariables = session.getChildren(session.getChild(element, "OutputVariables"), "OutputVariable");
        
        assertEquals(1, variables.size());
        assertEquals(1, dataObjectVariables.size());

        assertEquals("differentID", variables.toArray(new OutputVariable[0])[0].getVariableID());
        assertEquals("differentID", (dataObjectVariables.toArray(new Element[0])[0]).getAttribute("id"));
    }

    public void testSetRequiredFeatures() {
        ContentSelectionFeature feat = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createContentSelectionFeature("differentFeature");
        referencedPackage.setRequiredFeatures(Arrays.asList(feat));
        
        Collection<ContentSelectionFeature> features = referencedPackage.getRequiredFeatures();
        Collection<Element> dataObjectFeatures = session.getChildren(session.getChild(element, "RequiredContentSelection"), "Feature");
        
        assertEquals(1, features.size());
        assertEquals(1, dataObjectFeatures.size());

        assertEquals("differentFeature", features.toArray(new ContentSelectionFeature[0])[0].getFeature());
        assertEquals("differentFeature", (dataObjectFeatures.toArray(new Element[0])[0]).getTextContent());
    }

    public void testSetRequiredGroup() {
        referencedPackage.setRequiredGroup("differentGroup");

        assertEquals("differentGroup", referencedPackage.getRequiredGroup());
        assertEquals("differentGroup", session.getChildren((session.getChildren(element, "RequiredContentSelection").toArray(new Element[0])[0]), "Group").toArray(new Element[0])[0].getTextContent());
    }

    public void testSetRequirements() {
        Requirement req = SPISession.DEFAULT_INSTANCE.createRequirementFactory().createRequirement("differentID");
        referencedPackage.setRequirements(Arrays.asList(req));
        
        Collection<Requirement> reqs = referencedPackage.getRequirements();
        Collection<Element> dataObjectReqs = session.getChildren(session.getChild(element, "Requirements"), "Requirement");
        
        assertEquals(1, reqs.size());
        assertEquals(1, dataObjectReqs.size());

        assertEquals("differentID", reqs.toArray(new Requirement[0])[0].getRequirementID());
        assertEquals("differentID", (dataObjectReqs.toArray(new Element[0])[0]).getAttribute("id"));
    }

    public void testSetResultingChangeMaps() {
        ResultingChangeMap change = SPISession.DEFAULT_INSTANCE.createBaseFactory().createResultingChangeMap("differentRef");
        referencedPackage.setResultingChangeMaps(Arrays.asList(change));

        Collection<ResultingChangeMap> maps = referencedPackage.getResultingChangeMaps();
        Collection<Element> dataObjectMaps = session.getChildren(element, "ResultingChangeMap");
        
        assertEquals(1, maps.size());
        assertEquals(1, dataObjectMaps.size());

        assertEquals("differentRef", maps.toArray(new ResultingChangeMap[0])[0].getResourceReference());
        assertEquals("differentRef", (dataObjectMaps.toArray(new Element[0])[0]).getAttribute("resourceRef"));
    }

    public void testSetResultingResourceMaps() {
        ResultingResourceMap res = SPISession.DEFAULT_INSTANCE.createBaseFactory().createResultingResourceMap("differentRef");
        referencedPackage.setResultingResourceMaps(Arrays.asList(res));
        
        Collection<ResultingResourceMap> maps = referencedPackage.getResultingResourceMaps();
        Collection<Element> dataObjectMaps = session.getChildren(element, "ResultingResourceMap");
        
        assertEquals(1, maps.size());
        assertEquals(1, dataObjectMaps.size());

        assertEquals("differentRef", maps.toArray(new ResultingResourceMap[0])[0].getResourceReference());
        assertEquals("differentRef", (dataObjectMaps.toArray(new Element[0])[0]).getAttribute("resourceRef"));
    }

    public void testSetRequiredResourceMaps() {
        ResourceMap map = SPISession.DEFAULT_INSTANCE.createBaseFactory().createRequiredResourceMap("differentRef");
        referencedPackage.setRequiredResourceMaps(Arrays.asList(map));
        
        Collection<ResourceMap> maps = referencedPackage.getRequiredResourceMaps();
        Collection<Element> dataObjectMaps = session.getChildren(element, "RequiredResourceMap");
        
        assertEquals(1, maps.size());
        assertEquals(1, dataObjectMaps.size());

        assertEquals("differentRef", maps.toArray(new ResourceMap[0])[0].getResourceReference());
        assertEquals("differentRef", (dataObjectMaps.toArray(new Element[0])[0]).getAttribute("resourceRef"));
    }

    public void testSetWeight() {
        referencedPackage.setWeight(BigInteger.valueOf(3));

        assertEquals(3, referencedPackage.getWeight().intValue());
        assertEquals(3, new BigInteger(element.getAttribute("weight")).intValue());
    }

    public void testGetTranslatableStrings() {
        Properties props = ((SPIDataObject)referencedPackage).getTranslatableStrings();
        
        assertTrue(props.isEmpty());
    }
}
