/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.tests.common.spi;

import java.util.Arrays;
import java.util.Collection;
import java.util.Properties;

import junit.framework.TestCase;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ResultingChangeMap;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.SPISession;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.TranslatableString;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Relationship;
import org.w3c.dom.Element;

public class ResultingChangeMap_Test extends TestCase {
    ResultingChangeMap map;
    ResultingChangeMap clonedMap;
    Element element;
    SPISessionImpl session;

    public ResultingChangeMap_Test(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
        
        map = SPISession.DEFAULT_INSTANCE.createBaseFactory().createResultingChangeMap("resourceRef");

        Condition condition = SPISession.DEFAULT_INSTANCE.createConditionFactory().createCondition();
        condition.setDescription(new TranslatableString("desc", "descKey"));
        map.setCondition(condition);

        map.setForeignID("mapForeignID");
        map.setName("mapName");
        
        Properties props = new Properties();
        props.setProperty("key1", "value1");
        props.setProperty("key2", "value2");
        map.setProperties(props);
        
        Relationship rel1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createRelationship("map1ResourceRef", "map1Type");
        Relationship rel2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createRelationship("map2ResourceRef", "map2Type");
        map.setRelationships(Arrays.asList(rel1, rel2));

        clonedMap = (ResultingChangeMap)map.clone();
        element = ((SPIDataObject)map).getElement();
        session = (SPISessionImpl)SPISession.DEFAULT_INSTANCE;
    }

    public void testGetTranslatableStrings() {
        Properties props = ((SPIDataObject) map).getTranslatableStrings();
        Properties clonedProps = ((SPIDataObject) clonedMap).getTranslatableStrings();
        
        assertEquals(1, props.keySet().size());
        assertEquals(1, clonedProps.keySet().size());
        
        assertEquals("desc", props.get("descKey"));
        assertEquals("desc", clonedProps.get("descKey"));
    }

    public void testGetName() {
        assertEquals("mapName", map.getName());
        assertEquals("mapName", clonedMap.getName());
        assertEquals("mapName", session.getChildren(element, "Name").toArray(new Element[0])[0].getTextContent());
    }

    public void testGetProperties() {
        Properties testProps = map.getProperties();
        Properties testclonedProps = clonedMap.getProperties();
        Collection<Element> testDataObjectProps = session.getChildren(element, "Property");
        
        assertEquals("value1", testProps.get("key1"));
        assertEquals("value2", testProps.get("key2"));
        assertEquals("value1", testclonedProps.get("key1"));
        assertEquals("value2", testclonedProps.get("key2"));
        
        assertEquals(2, testDataObjectProps.size());
        
        for (int i = 0; i < testDataObjectProps.size(); i++) {
            Element prop = testDataObjectProps.toArray(new Element[0])[i];
            
            String value = session.getChildren(prop, "Value").toArray(new Element[0])[0].getTextContent();
            String propertyName = session.getChildren(prop, "PropertyName").toArray(new Element[0])[0].getTextContent();
            
            assertTrue(value + " wasn't found in the properties object", testProps.contains(value));
            assertTrue(propertyName + " wasn't found in the properties object", testProps.containsKey(propertyName));
            assertEquals(value, testProps.get(propertyName));
        }
    }

    public void testGetRelationships() {
        Collection<Relationship> rels = map.getRelationships();
        Collection<Relationship> clonedRels = clonedMap.getRelationships();
        Collection<Element> dataObjectRels = session.getChildren(element, "Relationship");
        
        assertEquals(2, rels.size());
        assertEquals(2, clonedRels.size());
        assertEquals(2, dataObjectRels.size());

        assertEquals("map1ResourceRef", rels.toArray(new Relationship[0])[0].getRelatedResourceReference());
        assertEquals("map1Type", rels.toArray(new Relationship[0])[0].getType());
        assertEquals("map2ResourceRef", rels.toArray(new Relationship[0])[1].getRelatedResourceReference());
        assertEquals("map2Type", rels.toArray(new Relationship[0])[1].getType());

        assertEquals("map1ResourceRef", clonedRels.toArray(new Relationship[0])[0].getRelatedResourceReference());
        assertEquals("map1Type", clonedRels.toArray(new Relationship[0])[0].getType());
        assertEquals("map2ResourceRef", clonedRels.toArray(new Relationship[0])[1].getRelatedResourceReference());
        assertEquals("map2Type", clonedRels.toArray(new Relationship[0])[1].getType());

        assertEquals("map1ResourceRef", (dataObjectRels.toArray(new Element[0])[0]).getAttribute("relatedResourceRef"));
        assertEquals("map1Type", (dataObjectRels.toArray(new Element[0])[0]).getAttribute("type"));
        assertEquals("map2ResourceRef", (dataObjectRels.toArray(new Element[0])[1]).getAttribute("relatedResourceRef"));
        assertEquals("map2Type", (dataObjectRels.toArray(new Element[0])[1]).getAttribute("type"));
    }

    public void testGetCondition() {
        Condition mapCondition = map.getCondition();
        Condition clonedMapCondition = clonedMap.getCondition();
        Element dataObjectMapCondition = session.getChildren(element, "Condition").toArray(new Element[0])[0];
        
        assertEquals("desc", mapCondition.getDescription().getValue());
        assertEquals("descKey", mapCondition.getDescription().getTranslationKey());

        assertEquals("desc", clonedMapCondition.getDescription().getValue());
        assertEquals("descKey", clonedMapCondition.getDescription().getTranslationKey());

        assertEquals("desc", session.getChildren(dataObjectMapCondition, "Description").toArray(new Element[0])[0].getTextContent());
        assertEquals("descKey", (session.getChildren(dataObjectMapCondition, "Description").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testGetForeignID() {
        assertEquals("mapForeignID", map.getForeignID());
        assertEquals("mapForeignID", clonedMap.getForeignID());
        assertEquals("mapForeignID", element.getAttribute("foreignId"));
    }

    public void testGetResourceReference() {
        assertEquals("resourceRef", map.getResourceReference());
        assertEquals("resourceRef", clonedMap.getResourceReference());
        assertEquals("resourceRef", element.getAttribute("resourceRef"));
    }

    public void testSetName() {
        map.setName("differentName");
        
        assertEquals("differentName", map.getName());
        assertEquals("differentName", session.getChildren(element, "Name").toArray(new Element[0])[0].getTextContent());
    }

    public void testSetProperties() {
        Properties properties = new Properties();
        properties.setProperty("testSetPropertiesKey", "testSetPropertiesValue");
        map.setProperties(properties);
        
        Properties testProps = map.getProperties();
        Collection<Element> testDataObjectProps = session.getChildren(element, "Property");
        
        assertEquals("testSetPropertiesValue", testProps.get("testSetPropertiesKey"));
        
        assertEquals(1, testDataObjectProps.size());
        
        for (int i = 0; i < testDataObjectProps.size(); i++) {
            Element prop = testDataObjectProps.toArray(new Element[0])[i];
            
            String value = session.getChildren(prop, "Value").toArray(new Element[0])[0].getTextContent();
            String propertyName = session.getChildren(prop, "PropertyName").toArray(new Element[0])[0].getTextContent();
            
            assertTrue(value + " wasn't found in the properties object", testProps.contains(value));
            assertTrue(propertyName + " wasn't found in the properties object", testProps.containsKey(propertyName));
            assertEquals(value, testProps.get(propertyName));
        }
    }

    public void testSetCondition() {
        Condition condition = SPISession.DEFAULT_INSTANCE.createConditionFactory().createCondition();
        condition.setDescription(new TranslatableString("differentDesc", "differentDescKey"));
        map.setCondition(condition);
        
        assertEquals("differentDesc", map.getCondition().getDescription().getValue());
        assertEquals("differentDescKey", map.getCondition().getDescription().getTranslationKey());
        
        Element conditionElement = session.getChildren(element, "Condition").toArray(new Element[0])[0];
        Element descElement = session.getChildren(conditionElement, "Description").toArray(new Element[0])[0];
        
        assertEquals("differentDesc", descElement.getTextContent());
        assertEquals("differentDescKey", descElement.getAttribute("translationKey"));
    }

    public void testSetRelationships() {
        Relationship rel = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createRelationship("differentResourceRef", "differentType");
        map.setRelationships(Arrays.asList(rel));

        Collection<Relationship> rels = map.getRelationships();
        Collection<Element> dataObjectRels = session.getChildren(element, "Relationship");
        
        assertEquals(1, rels.size());
        assertEquals(1, dataObjectRels.size());

        assertEquals("differentResourceRef", rels.toArray(new Relationship[0])[0].getRelatedResourceReference());
        assertEquals("differentType", rels.toArray(new Relationship[0])[0].getType());

        assertEquals("differentResourceRef", (dataObjectRels.toArray(new Element[0])[0]).getAttribute("relatedResourceRef"));
        assertEquals("differentType", (dataObjectRels.toArray(new Element[0])[0]).getAttribute("type"));
    }

    public void testSetForeignID() {
        map.setForeignID("differentID");
        
        assertEquals("differentID", map.getForeignID());
        assertEquals("differentID", element.getAttribute("foreignId"));
    }

    public void testSetResourceReference() {
        map.setResourceReference("differentRef");
        
        assertEquals("differentRef", map.getResourceReference());
        assertEquals("differentRef", element.getAttribute("resourceRef"));
        
        try {
            map.setResourceReference(null);
            
            fail("ResourceRef was set to null");
        }
        catch (Exception e) {
        }
    }
}
