/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.CompositeInstallable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ConfigurationUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ContentUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.InstallableUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.LocalizationUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;


/**
 * A solution package contains a deployment descriptor in addition to a package descriptor. The deployment descriptor describes 
 * the topology, selectability, inputs, requirements and conditions of the deployment. The deployment descriptor is associated 
 * with a package descriptor and refers to content files in that package descriptor.
 */
public interface DeploymentDescriptor extends Descriptor {
	
	/**
	 * Gets the ContentUnit in the descriptor.
	 * 
	 * The ContentUnit can be any one of the following:
	 * <li>{@link InstallableUnit}</li>
	 * <li>{@link ConfigurationUnit}</li>
	 * <li>{@link LocalizationUnit}</li><br><br>
	 * 
	 * @return The ContentUnit contained by the descriptor
	 */
	ContentUnit getContentUnit();
	
	/**
	 * Gets the CompositeInstallables in the descriptor.
	 * 
	 * A simple software deployment that uses a single artifact for each supported deployment operation MAY be described using an 
	 * SDD that defines a single atomic content elementInstallableUnit, ConfigurationUnit or LocalizationUnit.<br><br>
	 * A software deployment that requires multiple artifacts, aggregates other deployment packages or has selectable content MAY 
	 * be described using an SDD that defines one or more CompositeInstallable elements. Each CompositeInstallable MUST describe 
	 * one deployment lifecycle operation for the package.
	 * When one or more CompositeInstallable elements are defined, no InstallableUnit, ConfigurationUnit or LocalizationUnit elements 
	 * can be defined.
	 * 
	 * @return a collection of CompositeInstallable objects 
	 */
	Collection<CompositeInstallable> getCompositeInstallables();
	
	/**
	 * Gets the Resources in the descriptor.
	 * 
	 * A resource can participate by being required, created or modified by the deployment. A required resource MAY also play the role 
	 * of target resource, meaning that it can process artifacts to perform some portion of the deployment. The resources that actually 
	 * participate in a particular deployment are determined by the user inputs, selections and resource bindings provided during 
	 * that deployment.
	 * 
	 * @return a collection of Resource objects 
	 */
	Collection<Resource> getResources();
		
	/**
	 * Sets the ContentUnit in the descriptor.
	 * 
	 * * The ContentUnit can be any one of the following:
	 * <li>{@link InstallableUnit}</li>
	 * <li>{@link ConfigurationUnit}</li>
	 * <li>{@link LocalizationUnit}</li><br><br>
	 * 
	 * @param cu the ContentUnit to replace the existing ContentUnit contained by the descriptor
	 * @throws InvalidInputException if the parameter is null
	 */
	void setContentUnit(ContentUnit cu);
	
	/**
	 * Sets the CompositeInstallables in the descriptor.
	 * 
	 * A simple software deployment that uses a single artifact for each supported deployment operation MAY be described using an 
	 * SDD that defines a single atomic content elementInstallableUnit, ConfigurationUnit or LocalizationUnit.<br><br>
	 * A software deployment that requires multiple artifacts, aggregates other deployment packages or has selectable content MAY 
	 * be described using an SDD that defines one or more CompositeInstallable elements. Each CompositeInstallable MUST describe 
	 * one deployment lifecycle operation for the package.
	 * 
	 * @param ci a collection of CompositeInstallable objects to replace the existing CompositeInstallables contained by the descriptor
	 * @throws InvalidInputException if the parameter is null
	 */
	void setCompositeInstallables(Collection<CompositeInstallable> ci);
	
	/**
	 * Sets the Resources in the descriptor.
	 * 
	 * A resource can participate by being required, created or modified by the deployment. A required resource MAY also play the role 
	 * of target resource, meaning that it can process artifacts to perform some portion of the deployment. The resources that actually 
	 * participate in a particular deployment are determined by the user inputs, selections and resource bindings provided during 
	 * that deployment.
	 * 
	 * @param resources a collection of Resource objects to replace the existing Resources contained by the descriptor
	 * @throws InvalidInputException if the parameter is null
	 */
	void setResources(Collection<Resource> resources)throws InvalidInputException;
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	Object clone();
}