/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi;

/**
 * This complex type provides identity information for the package as a whole, as well as for content elements, which are portions 
 * of the package. Content elements are the InstallableUnit, LocalizationUnit, ConfigurationUnit, CompositeUnit and 
 * CompositeInstallable elements defined in the deployment descriptor.
 */
public interface Identity extends Nameable, Cloneable {
	
	/**
	 * Gets the Version.
	 * 
	 * This is a packaging version. In PackageIdentity, it is the version of the package as a whole. In content element identities, 
	 * this is the version of the unit of packaging represented by the content element. In either case, the SDD author MAY choose to 
	 * make this version correspond to the version of a resulting or changed resource, but it should not be confused with resource 
	 * versions.
	 * In the case of a base install, version MAY be the same as the top level resulting resource. In the case of a configuration 
	 * package, version SHOULD NOT be the same as the top level resulting resource.
	 * 
	 * @return The package or content element version.
	 */
	String getVersion();
	
	/**
	 * Gets the SoftwareID.
	 * 
	 * The software identified by softwareID is the software whose deployment is described by the SDD. When the manufacturer 
	 * maintains software identifiers within a sales and distribution system, the softwareID SHOULD correspond to an identifier 
	 * for the software within that system. If a format for software identifiers is not pre-existing within the manufacturers 
	 * systems, a UUID SHOULD be used for softwareID. When a UUID is used, it MUST be unique within the domain in which the 
	 * described software is used.
	 * 
	 * @return A manufacturers identification number for the software created or updated by the package or content element.
	 */
	String getSoftwareID();
	
	/**
	 * Gets the BuildID.
	 * 
	 * The buildID attribute is an identifier provided by the manufacturer and meaningful to developers that can be used to identify 
	 * a build of the defining element. This information MUST correspond with information known in the manufacturers build environment.
	 * It is traditionally used during problem determination to allow maintainers of the software to determine the specifics of package 
	 * creation. Inclusion of buildID in the SDD allows the end user to provide this information to package maintainers, enabling them 
	 * to correlate the deployed software with a particular known build of the software.
	 * 
	 * @return A string that identifies the build of the package or package element.
	 */
	String getBuildID();
	
	/**
	 * Gets the ManufacturerInformation.
	 * 
	 * Information about the manufacturer of the package or content element.
	 * 
	 * @return A ManufacturerInformation object that contains information about the manufacturer of the package or content element.
	 */
	ManufacturerInformation getManufacturerInformation();
	
	/**
	 * Gets the MaintenanceInformation.
	 * 
	 * This is used when the package or content element describes the deployment of maintenance. 
	 * 
	 * @return A MaintenanceInformation object that contains information about package or content element content used when the package contains maintenance.
	 */
	MaintenanceInformation getMaintenanceInformation();
	
	/**
	 * Sets the Version.
	 * 
	 * This is a packaging version. In PackageIdentity, it is the version of the package as a whole. In content element identities, 
	 * this is the version of the unit of packaging represented by the content element. In either case, the SDD author MAY choose to 
	 * make this version correspond to the version of a resulting or changed resource, but it should not be confused with resource 
	 * versions.
	 * In the case of a base install, version MAY be the same as the top level resulting resource. In the case of a configuration 
	 * package, version SHOULD NOT be the same as the top level resulting resource.
	 * 
	 * @param version The package or content element version.
	 */
	void setVersion(String version);
	
	/**
	 * Sets the SoftwareID.
	 * 
	 * The software identified by softwareID is the software whose deployment is described by the SDD. When the manufacturer 
	 * maintains software identifiers within a sales and distribution system, the softwareID SHOULD correspond to an identifier 
	 * for the software within that system. If a format for software identifiers is not pre-existing within the manufacturers 
	 * systems, a UUID SHOULD be used for softwareID. When a UUID is used, it MUST be unique within the domain in which the 
	 * described software is used.
	 * 
	 * @param softwareID A manufacturers identification number for the software created or updated by the package or content element.
	 */
	void setSoftwareID(String softwareID);
	
	/**
	 * Sets the BuildID.
	 * 
	 * The buildID attribute is an identifier provided by the manufacturer and meaningful to developers that can be used to identify 
	 * a build of the defining element. This information MUST correspond with information known in the manufacturers build environment.
	 * It is traditionally used during problem determination to allow maintainers of the software to determine the specifics of package 
	 * creation. Inclusion of buildID in the SDD allows the end user to provide this information to package maintainers, enabling them 
	 * to correlate the deployed software with a particular known build of the software.
	 * 
	 * @param buildID A string that identifies the build of the package or package element.
	 */
	void setBuildID(String buildID);
	
	/**
	 * Sets the ManufacturerInformation.
	 * 
	 * Information about the manufacturer of the package or content element. 
	 * 
	 * @param information A ManufacturerInformation object that contains information about the manufacturer of the package or content element.
	 */
	void setManufacturerInformation(ManufacturerInformation information);
	
	/**
	 * Sets the MaintenanceInformation.
	 * 
	 * This is used when the package or content element describes the deployment of maintenance. 
	 * 
	 * @param information A MaintenanceInformation object that contains information about package or content element content used when the package contains maintenance.
	 */
	void setMaintenanceInformation(MaintenanceInformation information);
	
	/**
	 * Creates a copy of this object. 
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();
}