/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi;

import java.math.BigInteger;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.xmldsig.Digest;

/**
 * A software package includes one or more content files. ContentType defines the properties of a content file included in the package 
 * descriptor. Content defined in the package descriptor as part of the software package does not need to be physically co-located. 
 * Each element MUST be in a location that can be identified by a URI. The pathname attribute of each content file defines a URI for 
 * accessing the file. Characteristics of the content filessuch as their length, purpose and character encodingMAY be declared in 
 * the package descriptor.
 */
public interface PackageContent extends Digest {
	
	/**
	 * Gets the ContentID.
	 * 
	 * This is the identifier for the content that is used as a reference in artifact elements in the deployment descriptor.<br><br>
	 * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
	 * 
	 * @return An identifier used in deployment descriptors to refer to the file definition in the associated package descriptor.
	 */
	String getContentID();
	
	/**
	 * Gets the Pathname.
	 * 
	 * Pathname is used to access content in the package. The path of the file MUST be a URI that specifies an absolute path or a path 
	 * relative to the location of the package descriptor. It MUST include the file name.
	 * 
	 * @return The absolute or relative path of the content file including the file name.
	 */
	String getPathname();
	
	/**
	 * Gets the ContentPurpose.
	 * 
	 * The purpose attribute enables the PackageDescriptor author to associate a classification with a file. The classification 
	 * identifies the file as having a specific purpose. ContentPurposeType defines a union of SDDContentPurposeType with xsd:NCName. 
	 * The purpose value MAY be chosen from one of the following values enumerated in SDDContentPurposeType or be a valid NCName value 
	 * provided by the SDD author. If purpose is not specified, the default value is content.<br><br>
	 * Enumerated values for purpose are: 
	 * <li>readMe</li>
	 * <li>endUserLicenseAgreement</li>
	 * <li>responseFile</li>
	 * <li>deploymentDescriptor</li>
	 * <li>packageDescriptor</li>
	 * <li>descriptorLanguageBundle</li>
	 * <li>content</li><br><br>
	 * 
	 * @return The purpose classification associated with a file.
	 */
	String getContentPurpose();
	
	/**
	 * Gets the Character Encoding.
	 * 
	 * This attribute need only be used for files that a run-time is required to render. Common charEncoding values include ASCII, 
	 * UTF-8, UTF-16 and Shift_JIS. 
	 * 
	 * @return The character encoding of the contents of the file.
	 */
	String getCharEncoding();
	
	/**
	 * Gets the length.
	 * 
	 * The file length MAY be used for simple file verification. 
	 * 
	 * @return Length of the file in bytes.
	 */
	BigInteger getLength();
		
	
	/**
	 * Sets the Character Encoding.
	 * 
	 * This attribute need only be used for files that a run-time is required to render. Common charEncoding values include ASCII, 
	 * UTF-8, UTF-16 and Shift_JIS. 
	 * 
	 * @param charEncoding The character encoding of the contents of the file.
	 */
	void setCharEncoding(String charEncoding);
	
	/**
	 * Sets the ContentID.
	 * 
	 * This is the identifier for the content that is used as a reference in artifact elements in the deployment descriptor.<br><br>
	 * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
	 * 
	 * @param contentID An identifier used in deployment descriptors to refer to the file definition in the associated package descriptor.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setContentID(String contentID);
	
	/**
	 * Sets the ContentPurpose.
	 * 
	 * The purpose attribute enables the PackageDescriptor author to associate a classification with a file. The classification 
	 * identifies the file as having a specific purpose. ContentPurposeType defines a union of SDDContentPurposeType with xsd:NCName. 
	 * The purpose value MAY be chosen from one of the following values enumerated in SDDContentPurposeType or be a valid NCName value 
	 * provided by the SDD author. If purpose is not specified, the default value is content.<br><br>
	 * Enumerated values for purpose are: 
	 * <li>readMe</li>
	 * <li>endUserLicenseAgreement</li>
	 * <li>responseFile</li>
	 * <li>deploymentDescriptor</li>
	 * <li>packageDescriptor</li>
	 * <li>descriptorLanguageBundle</li>
	 * <li>content</li><br><br>
	 * 
	 * @param contentPurpose The purpose classification associated with a file.
	 */
	void setContentPurpose(String contentPurpose);
	
	/**
	 * Sets the length.
	 * 
	 * The file length MAY be used for simple file verification. 
	 * 
	 * @param length The length of the file in bytes.
	 */
	void setLength(BigInteger length);
	
	/**
	 * Sets the Pathname.
	 * 
	 * Pathname is used to access content in the package. The path of the file MUST be a URI that specifies an absolute path or a path 
	 * relative to the location of the package descriptor. It MUST include the file name.
	 * 
	 * @param pathname The absolute or relative path of the content file including the file name.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setPathname(String pathname);

	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	Object clone();
}
