/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi;

import java.util.Collection;
import java.util.Properties;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Relationship;

/**
 * ResultingChangeMapType is very similar to ResultingResourceMapType. It defines an element type that maps resources that are changed 
 * by deployment of the referenced SDD to a resource in the referencing SDD. In addition to identifying the two resources that MUST 
 * resolve to the same resource instance, the resulting change map allows characteristics of the modified resource to be exposed. 
 * There may be constraints defined on the mapped resource in the referencing SDD or any referenced SDD in the hierarchy of SDDs. 
 * These constraints can be evaluated by comparing the constraint to the exposed characteristics defined in the resulting change map. 
 * The resulting change map MUST expose sufficient characteristics of the resulting change to support successful evaluation of 
 * constraints on that resource. 
 * <ul>For example, say that the SDD defines a resource with id=OS in its topology. The solution can work with Windows or Linux. 
 * Linux is configured by a referenced SDD defined in a Requisites element. The SDD will contain Requirements and/or Conditions that have 
 * alternatives for Windows and for Linux. All constraints on the modified characteristics of Linux must be satisfied by a resource 
 * characteristic exposed in the ResultingChangeMap element of the Requisite element that points to the SDD that configures Linux.</ul>
 */
public interface ResultingChangeMap extends ResourceMap {
	
    /**
     * Gets the Condition.
     * 
     * A Condition is used when the resource mapped from the external package will be changed only when certain conditions exist in 
     * the deployment environment.
     * 
     * @return A Condition object that determines if the resulting change definition is relevant to a particular deployment.
     */
    Condition getCondition();
    
    /**
     * Gets the name.
     * 
     * The Name of the resource that is modified by the referenced SDD is defined here to assist with identifying the resource 
     * instance that is changed. It is not an indication that the resource name itself is modified by the referenced SDD. If resource 
     * characteristics defined in the topology of any SDD defining a resource mapped to the changed resource are sufficient to identify 
     * the resource, then Name SHOULD NOT be defined in the ResultingChangeMap.
     * 
     * @return The name of the modified resource.
     */
    String getName();
    
    /**
     * Gets the Properties.
     * 
     * A modified property MUST be exposed in a ResultingChangeMap if it is not defined elsewhere and there are property constraints 
     * on the modified property. Defined elsewhere means defined in the topology of the referencing SDD or in the topology of any 
     * other referenced SDD for a resource that is also mapped to the same resource. Constraints on the modified property means a 
     * property constraint that applies to the particular instantiation of the resource that is modified by the referenced SDD, for 
     * example a constraint that needs to successfully map to the referenced SDD for the referenced SDD to be used in a particular 
     * deployment.
     * 
     * @return A Properties object that contains modified property of the resource.
     */
    Properties getProperties();
    
    /**
     * Gets the Relationship.
     * 
     * Relationship elements SHOULD be included to identify relationships that will exist after the application of the referenced 
     * package.<br><br>
     * Relationships that need to be known by the aggregate MUST be mapped. Relationships need to be known when they are referred to 
     * in one or more resource constraints.
     * 
     * @return A collection of Relationships that will exist after the change is applied to the resource.
     */
    Collection<Relationship> getRelationships();
    
    /**
     * Sets the Condition.
     * 
     * A Condition is used when the resource mapped from the external package will be changed only when certain conditions exist in 
     * the deployment environment.
     * 
     * @param condition A condition that determines if the resulting change definition is relevant to a particular deployment.
     */
    void setCondition(Condition condition);
    
    /**
     * Sets the name.
     * 
     * The Name of the resource that is modified by the referenced SDD is defined here to assist with identifying the resource 
     * instance that is changed. It is not an indication that the resource name itself is modified by the referenced SDD. If resource 
     * characteristics defined in the topology of any SDD defining a resource mapped to the changed resource are sufficient to identify 
     * the resource, then Name SHOULD NOT be defined in the ResultingChangeMap.
     * 
     * @param name The name of the modified resource.
     */
    void setName(String name);
    
    /**
     * Sets the Properties.
     * 
     * A modified property MUST be exposed in a ResultingChangeMap if it is not defined elsewhere and there are property constraints 
     * on the modified property. Defined elsewhere means defined in the topology of the referencing SDD or in the topology of any 
     * other referenced SDD for a resource that is also mapped to the same resource. Constraints on the modified property means a 
     * property constraint that applies to the particular instantiation of the resource that is modified by the referenced SDD, for 
     * example a constraint that needs to successfully map to the referenced SDD for the referenced SDD to be used in a particular 
     * deployment.
     * 
     * @param props A modified property of the resource.
     */
    void setProperties(Properties props);
    
    /**
     * Sets the Relationships.
     * 
     * Relationship elements SHOULD be included to identify relationships that will exist after the application of the referenced 
     * package.<br><br>
     * Relationships that need to be known by the aggregate MUST be mapped. Relationships need to be known when they are referred to 
     * in one or more resource constraints.
     * 
     * @param A collection of Relationships that will exist after the change is applied to the resource.
     */
    void setRelationships(Collection<Relationship> relationships);

	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();
}
