/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi;

import java.util.Collection;
import java.util.Properties;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Relationship;

/**
 * ResultingResourceMapType defines an element type that maps resources that result from deployment of the referenced SDD to a 
 * resource in the referencing SDD. In addition to identifying the two resources that MUST resolve to the same resource instance, 
 * the resulting resource map allows characteristics of the resulting resource to be exposed. There may be constraints defined on 
 * the mapped resource in the referencing SDD or any referenced SDD in the hierarchy of SDDs. These constraints can be evaluated by 
 * comparing the constraint to the exposed characteristics defined in the resulting resource map. The resulting resource map MUST 
 * expose sufficient characteristics of the resulting resource to support successful evaluation of constraints on that resource. 
 * <ul>For example, say that the SDD defines a resource with id=Database in its topology. The solution can work with Database 
 * Product A or Database Product B. Database Product A is created by a referenced SDD defined in a Requisites element. The SDD will 
 * contain Requirements and/or Conditions that have alternatives for each of the database products. All constraints on the Database 
 * resource that apply to Database Product A must be satisfied by a resource characteristic exposed in the ResultingResourceMap element 
 * of the Requisite element that points to the SDD that deploys Database Product A. 
 * 
 */
public interface ResultingResourceMap extends ResourceMap {
	
    /**
     * Gets the Condition.
     * 
     * A Condition is used when the resulting resource will be created by the referenced package only when certain conditions exist 
     * in the deployment environment.
     *  
     * @return A Condition object that determines if the resulting resource definition is relevant to a particular deployment.
     */
    Condition getCondition();
    
    /**
     * Gets the name.
     * 
     * The Name of the resulting resource created or updated by the referenced SDD MUST be defined if it is not defined elsewhere and 
     * there are constraints on this resource that contain a Name element. Defined elsewhere means defined in the topology of the 
     * referencing SDD or in the topology of any other referenced SDD for a resource that is also mapped to the same resource. 
     * Constraints on this resource means a constraint that applies to the particular instantiation of the resource that is created 
     * or updated by the referenced SDD, for example a constraint that needs to successfully map to the referenced SDD for the 
     * referenced SDD to be used in a particular deployment.
     * 
     * @return The name of the resource created or updated by the referenced SDD.
     */
    String getName();
    
    /**
     * Gets the version.
     * 
     * The Version of the resulting resource created or updated by the referenced SDD MUST be defined if it is not defined elsewhere 
     * and there are version constraints defined on this resource. (See the usage note for Name above for a definition of defined 
     * elsewhere.)
     * 
     * @return The version of the resource created or updated by the referenced SDD.
     */
    String getVersion();
    
    /**
     * Gets the fix name.
     * 
     * One or more names of fixes to the resulting resource created or updated by the referenced SDD MUST be defined if they are not 
     * defined elsewhere and there are version constraints defined on this resource that include fix names. (See the usage note for 
     * Name above for a definition of defined elsewhere.)
     * 
     * @return A collection of names of fixes to the mapped resource that are created by the referenced SDD.
     */
    Collection<String> getFixNames();
    
    /**
     * Gets the Properties.
     * 
     * A Property of the resulting resource created or updated by the referenced SDD MUST be defined if it is not defined elsewhere and 
     * there are property constraints on this property. (See the usage note for Name above for a definition of defined elsewhere.)
     * 
     * @return Properties set when the mapped resource is created or updated by the referenced SDD.
     */
    Properties getProperties();
    
    /**
     * Gets the Relationship.
     * 
     * Any number of Relationship elements can be included to identify relationships that will exist after applying the referenced 
     * package. 
     * 
     * @return A collection of Relationships that will exist after creating or updating the resource.
     */
    Collection<Relationship> getRelationships();
    
    /**
     * Sets the Condition.
     * 
     * A Condition is used when the resulting resource will be created by the referenced package only when certain conditions exist 
     * in the deployment environment.
     *  
     * @param condition A condition that determines if the resulting resource definition is relevant to a particular deployment.
     */
    void setCondition(Condition condition);
    
    /**
     * Sets the name.
     * 
     * The Name of the resulting resource created or updated by the referenced SDD MUST be defined if it is not defined elsewhere and 
     * there are constraints on this resource that contain a Name element. Defined elsewhere means defined in the topology of the 
     * referencing SDD or in the topology of any other referenced SDD for a resource that is also mapped to the same resource. 
     * Constraints on this resource means a constraint that applies to the particular instantiation of the resource that is created 
     * or updated by the referenced SDD, for example a constraint that needs to successfully map to the referenced SDD for the 
     * referenced SDD to be used in a particular deployment.
     * 
     * @param name The name of the resource created or updated by the referenced SDD.
     */
    void setName(String name);
    
    /**
     * Sets the version.
     * 
     * The Version of the resulting resource created or updated by the referenced SDD MUST be defined if it is not defined elsewhere 
     * and there are version constraints defined on this resource. (See the usage note for Name above for a definition of defined 
     * elsewhere.)
     * 
     * @param version The version of the resource created or updated by the referenced SDD.
     */
    void setVersion(String version);
    
    /**
     * Sets the fix names.
     * 
     * One or more names of fixes to the resulting resource created or updated by the referenced SDD MUST be defined if they are not 
     * defined elsewhere and there are version constraints defined on this resource that include fix names. (See the usage note for 
     * Name above for a definition of defined elsewhere.)
     * 
     * @param names A collection of names of fixes to the mapped resource that are created by the referenced SDD.
     */
    void setFixNames(Collection<String> names);
    
    /**
     * Sets the properties.
     * 
     * A Property of the resulting resource created or updated by the referenced SDD MUST be defined if it is not defined elsewhere and 
     * there are property constraints on this property. (See the usage note for Name above for a definition of defined elsewhere.)
     * 
     * @param props Properties set when the mapped resource is created or updated by the referenced SDD.
     */
    void setProperties(Properties props);
    
    /**
     * Sets the Relationships.
     * 
     * Any number of Relationship elements can be included to identify relationships that will exist after applying the referenced 
     * package. 
     * 
     * @param relationships A collection of Relationships that will exist after creating or updating the resource.
     */
    void setRelationships(Collection<Relationship> relationships);
    
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();
}
