/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.DisplayNameable;

/**
 * ConditionType allows expression of the particular resource characteristics that must be true for the condition to be considered met. 
 * These are resource characteristics that may vary from one particular deployment to another. 
 * <ul>For example, one deployment using the SDD might use one version of an application server and a different deployment might use a 
 * different version. The differences in the version might be great enough to:<br><br>
 * <li>select among content elements.</li>
 * <ul>For example, one content element has an artifact for a Web application that works in a particular version and a different 
 * content element has an artifact for a later version of the same Web application.</ul>
 * <li>select among variable values.</li> 
 * <ul>For example, the default installation path on one operating system may be different from the default install path on another 
 * operating system.</ul>
 * <li>select among completion actions.</li>
 * <ul>For example, a reboot may be required when deploying on one operating system but not another.</ul>
 * </ul>
 */
public interface Condition extends DisplayNameable {

    /**
     * Gets the AlternativeConditions.
     * 
     * When a condition can be satisfied in multiple ways, two or more Alternative elements are defined.<br><br>
     * As a convenience for tooling that produces SDDs, it is also possible to define a single Alternative. This is semantically 
     * identical to directly defining ResourceConstraints.<br><br>
     * To meet a condition, at least one of the specified Alternatives must be satisfied.
     *  
     * @return A collection of AlternativeCondition.
     */
    Collection<AlternativeCondition> getAlternativeConditions();
    
    /**
     * Gets the operations.
     * 
     * In a singleton atomic content element, a condition MAY be associated with application of one or more artifacts. The association 
     * is made by setting the operation attribute to the operations associated with those artifacts.<br><br>
     * Conditions defined for CompositeInstallable and for atomic content elements defined within a CompositeInstallable SHOULD NOT 
     * define operation. If the operation is defined for a CompositeInstallable Condition, it MUST be set to the operation defined in 
     * the CompositeInstallables operation attribute. If operation is defined for an atomic content elements Condition, it MUST be 
     * set to the operation associated with the single artifact defined by the atomic content element.<br><br>
     * When operation is not specified, the condition applies to the processing of all artifacts.
     *  
     * @return A collection of operations.
     */
    Collection<String> getOperations();
    
    /**
     * Gets the ConditionalResourceConstraints.
     * 
     * When a condition can be satisfied in only one way, constraints MAY be defined directly under Condition or in a single 
     * Alternative element.<br><br>
     * Constraints are defined using a sequence of ResourceConstraints. Every constraint in the sequence must be met for the condition 
     * to be met.
     *  
     * @return A collection of ConditionalResourceConstraint.
     */
    Collection<ConditionalResourceConstraint> getConditionalResourceConstraints();
    
    /**
     * Sets the AlternativeConditions.
     * 
     * When a condition can be satisfied in multiple ways, two or more Alternative elements are defined.<br><br>
     * As a convenience for tooling that produces SDDs, it is also possible to define a single Alternative. This is semantically 
     * identical to directly defining ResourceConstraints.<br><br>
     * To meet a condition, at least one of the specified Alternatives must be satisfied.
     * 
     * @param alternativeConditions A collection of AlternativeCondition.
     */
    void setAlternativeConditions(Collection<AlternativeCondition> alternativeConditions);
    
    /**
     * Gets the operations.
     * 
     * In a singleton atomic content element, a condition MAY be associated with application of one or more artifacts. The association 
     * is made by setting the operation attribute to the operations associated with those artifacts.<br><br>
     * Conditions defined for CompositeInstallable and for atomic content elements defined within a CompositeInstallable SHOULD NOT 
     * define operation. If the operation is defined for a CompositeInstallable Condition, it MUST be set to the operation defined in 
     * the CompositeInstallables operation attribute. If operation is defined for an atomic content elements Condition, it MUST be 
     * set to the operation associated with the single artifact defined by the atomic content element.<br><br>
     * When operation is not specified, the condition applies to the processing of all artifacts.
     * 
     * @param operations A collection of operations.
     */
    void setOperations(Collection<String> operations);
    
    /**
     * Gets the ConditionalResourceConstraints.
     * 
     * When a condition can be satisfied in only one way, constraints MAY be defined directly under Condition or in a single 
     * Alternative element.<br><br>
     * Constraints are defined using a sequence of ResourceConstraints. Every constraint in the sequence must be met for the condition 
     * to be met.
     * 
     * @param crc A collection of ConditionalResourceConstraint.
     */
    void setConditionalResourceConstraints(Collection<ConditionalResourceConstraint> crc);
    
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
    Object clone();
}
