/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.DisplayNameable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;


/**
 * For some deployments certain completion actions such as restart and logoff are required before a deployment operation 
 * using a particular content element can be considered complete. The CompletionType elements enable the SDD author to 
 * indicate either that one of these actions is required or that one of these actions will be performed by the associated artifact.
 */
public interface Completion extends DisplayNameable {
	
	/**
	 * Gets the completion type.
	 * 
	 * This is the completion action that applies when conditions defined in ResourceConstraint are met. Allowed values defined in 
	 * CompletionTypeNameType are:<br><br>
	 * <li>restartRequiredImmediately: A system restart is required before the deployment operation is considered complete and the 
	 * artifact associated with the operation does not perform the restart. The restart MUST happen before further deployment actions 
	 * are taken.</li>
	 * <li>restartRequiredBeforeUse: A system restart is required before the deployment operation is considered complete and the 
	 * artifact associated with the operation does not perform this action. The restart MUST happen before the associated resources 
	 * are used.</li>
	 * <li>restartOccurs: The artifact associated with the lifecycle operation will initiate a system restart.</li>
	 * <li>logoffRequired: A logoff and logon to the user account is required before the deployment operation is considered complete 
	 * and the artifact associated with the operation does not perform this action. The logoff and logon MUST happen before the 
	 * operation can be considered complete.</li>
	 * 
	 * @return The type of the completion action.
	 */
	String getCompletionType();
	
	/**
	 * Gets the completion resource reference.
	 * 
	 * This will often be the resource named as the target resource for the defining content element.<br><br>
	 * The value MUST reference the id of a resource element in Topology.
	 *
	 * @return The resource where the completion action will be executed.
	 */
	String getCompletionResourceReference();
	
	/**
	 * Gets the completion operations.
	 * 
	 * A completion action is associated with the processing of one artifact by setting operation to the operation associated with 
	 * that artifact. The element that defines the Completion MUST also define an artifact associated with the operation defined for 
	 * the Completion element.
	 * 
	 * @return The collection of completion actions that are associated with the processing of a particular artifact.
	 */
	Collection<String> getCompletionOperations();

	/**
	 * Gets the condition.
	 * 
	 * Conditions specified on resource characteristics determine if the completion action applies. If the conditions are met, the 
	 * action applies. If not met, then the action is not needed. Unmet conditions are not considered a failure. When no conditions 
	 * are defined, the action always applies.
	 * 
	 * @return The Condition that determined when the completion action will be used.
	 */
	Condition getCondition();

	
	/**
	 * Sets the completion type.
	 * 
	 * This is the completion action that applies when conditions defined in ResourceConstraint are met. Allowed values defined in 
	 * CompletionTypeNameType are:
	 * <li>restartRequiredImmediately: A system restart is required before the deployment operation is considered complete and the 
	 * artifact associated with the operation does not perform the restart. The restart MUST happen before further deployment actions 
	 * are taken.</li>
	 * <li>restartRequiredBeforeUse: A system restart is required before the deployment operation is considered complete and the artifact 
	 * associated with the operation does not perform this action. The restart MUST happen before the associated resources are used.</li>
	 * <li>restartOccurs: The artifact associated with the lifecycle operation will initiate a system restart.</li>
	 * <li>logoffRequired: A logoff and logon to the user account is required before the deployment operation is considered complete and 
	 * the artifact associated with the operation does not perform this action. The logoff and logon MUST happen before the operation can 
	 * be considered complete.</li>
	 * 
	 * @param completionType The type of the completion action.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setCompletionType(String completionType);
	
	/**
	 * Sets the completion resource reference.
	 * 
	 * This will often be the resource named as the target resource for the defining content element.<br><br>
	 * The value MUST reference the id of a resource element in Topology.
	 *
	 * @param completionResourceReference The resource where the completion action will be executed.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setCompletionResourceReference(String completionResourceReference);

	/**
	 * Sets the completion operations.
	 * 
	 * A completion action is associated with the processing of one artifact by setting operation to the operation associated with 
	 * that artifact. The element that defines the Completion MUST also define an artifact associated with the operation defined for 
	 * the Completion element.
	 * 
	 * @param operation The collection of completion actions that are associated with the processing of a particular artifact.
	 */
	void setCompletionOperations(Collection<String> operations);
	
	/**
	 * Sets the condition.
	 * 
	 * Conditions specified on resource characteristics determine if the completion action applies. If the conditions are met, the 
	 * action applies. If not met, then the action is not needed. Unmet conditions are not considered a failure. When no conditions 
	 * are defined, the action always applies.
	 * 
	 * @param condition The Condition that determines when the completion action will be used.
	 */
	void setCondition(Condition condition);
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	Object clone();
}
