/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Descriptor;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Identity;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.ConditionalDerivedVariable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.DerivedVariable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.ParameterGroup;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.ResourceProperty;

/**
 * A base interface for the {@link InstallableUnit}, {@link LocalizationUnit} and
 * {@link ConfigurationUnit} interfaces.
 *
 */
public interface ContentUnit {
	/**
	 * Gets the content unit ID.
	 * 
	 * Returns the content unit ID, which is a String that conforms to the ID attribute
	 * spec, as described here: http://www.w3.org/TR/xmlschema-2/#ID
	 * 
	 * It's an NCNAME, which is described here:
	 * 	http://www.w3.org/TR/1999/REC-xml-names-19990114/#NT-NCName
	 * 
	 * 
	 * @return The content unit ID.
	 */
	String getContentUnitID();
	
	
	/**
	 * Gets the identity data.
	 * 
	 * @return The Identity object associated with the content unit.
	 */
	Identity getIdentityData();
	
	/**
	 * Gets the DerivedVariable.
	 * 
	 * @return The DerivedVariable objects associated with the content unit.
	 */
	Collection<DerivedVariable> getDerivedVariables();

	/**
	 * Gets the ConditionalDerivedVariable.
	 * 
	 * @return The ConditionalDerivedVariable objects associated with the content unit.
	 */
	Collection<ConditionalDerivedVariable> getConditionalDerivedVariables();
	
	/**
	 * Gets the ResourceProperties.
	 * 
	 * @return The ResourceProperty objects associated with the content unit.
	 */
	Collection<ResourceProperty> getResourceProperties();
	
	/**
	 * Gets the ParameterGroup.
	 * 
	 * @return The ParameterGroup objects associated with the content unit.
	 */
	Collection<ParameterGroup> getParameterGroups();
	
	/**
	 * Sets the content unit ID.
	 * 
	 * @param contentUnitID The identifier that uniquely describes the content.
	 * unit within its containing {@link Descriptor}
	 * @throws InvalidInputException if the parameter is null
	 */
	void setContentUnitID(String contentUnitID);
	
	/**
	 * Sets the identity data.
	 * 
	 * @param contentUnitIdentity The Identity that uniquely describes the content.
	 * unit within its containing {@link Descriptor}
	 * 
	 */
	public void setIdentityData(Identity contentUnitIdentity);
	
	/**
	 * Sets the derived variables.
	 * 
	 * @param derivedVariables The DerivedVariable objects associated with the content unit.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setDerivedVariables(Collection<DerivedVariable> derivedVariables);
	
	/**
	 * Sets the conditional derived variables.
	 * 
	 * @param conditionalDerivedVariables The ConditionalDerivedVariable objects associated with the content unit.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setConditionalDerivedVariables(Collection<ConditionalDerivedVariable> conditionalDerivedVariables);
	
	/**
	 * Sets the ResourceProperties.
	 * 
	 * @param newResourceProperties The ResourceProperty objects associated with the content unit.
	 *  
	 */
	void setResourceProperties(Collection<ResourceProperty> newResourceProperties);
	
	/**
	 * Sets the ParameterGroup.
	 * 
	 * @param newParameterGroups The ParameterGroup objects associated with the content unit.
	 */
	void setParameterGroups(Collection<ParameterGroup> newParameterGroups);
}
