/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.artifact.Artifact;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requiredbase.RequiredBase;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.Requirement;


/**
 * The InstallUnit interface consolidates content and function common to
 * the {@link InstallableUnit} and {@link LocalizationUnit} interfaces.
 *
 */
public interface InstallUnit extends ContentUnit {
	/**
     * Gets the target resource reference.
     * 
     * The targetResourceRef attribute identifies the resource that will process the InstallableUnits artifacts.
     * The resources created or modified by artifact processing are frequently, but not necessarily, hosted by 
     * the target resource. 
     * 
     * This value MUST match an id of a resource element in Topology. The target may be a 
     * resource that has not yet been created. In this case, there is a dependency on the complete installation 
     * of the target resource prior to applying the InstallableUnit. This dependency MUST be represented in a 
     * Dependency element within Requirements that apply to the InstallableUnit.  
     *   
     * @return Reference to the resource that can process the InstallableUnits artifacts.
     */
	String getTargetResourceReference();
	
	/**
     * Gets the languages.
     * 
     * When translated materials are deployed by the InstallableUnits artifacts, the languages of the 
     * translations are listed in Languages.  
     *   
     * @return A collection of languages supported by the InstallableUnit.
     */
	Collection<Language> getLanguages();
	
	/**
     * Gets the requirements.
     * 
     * Requirements specified in an InstallableUnit identify requirements that must be met prior to 
     * successful processing of the InstallableUnits artifacts. 
     *   
     * @return A collection of requirements that must be met prior to successful processing of the 
     * InstallableUnits artifacts.
     */
	Collection<Requirement> getRequirements();
	
	/**
     * Gets the required base.
     * 
     * When an InstallableUnit can be used to update resources, the RequiredBase element 
     * identifies the resources that can be updated. 
     *   
     * @return A resource that will be updated when the InstallableUnits UpdateArtifact is processed.
     */
	RequiredBase getRequiredBase();
	
	/**
     * Gets the resulting resources.
     * 
     * An InstallableUnits ResultingResource element identifies the resources in Topology that will be installed or 
     * updated when the InstallableUnits artifacts are processed.
     *   
     * @return A collection of resources that will be installed or updated by processing the InstallableUnits artifacts.
     */
	Collection<ResultingResource> getResultingResources();
	
	/**
	 * Gets the install artifact.
     * 
     * When the InstallableUnit is a singleton defined outside of a CompositeInstallable, it MUST define at least one 
     * artifact element and MAY define one of each type of artifact element allowed for its type. The inclusion of an 
     * artifact element in a singleton InstallableUnit implies support for the associated operation.
     * 
     * When the InstallableUnit is defined within a CompositeInstallable, it MUST define exactly one artifact. 
     * The artifact defined MAY be any artifact allowed in an InstallableUnit and it MUST support the single top level 
     * to be the same as the one defined by the CompositeInstallable.
     * 
     * For example, an update of a resource may be required to support an install of the overall solution, in which case 
     * the InstallableUnit would define an UpdateArtifact to support the top level install operation. 
     * 
	 * @return The install Artifact for the install unit.
	 */
	Artifact getInstallArtifact();
	
	/**
	 * Gets the uninstall artifact.
	 * 
	 * 
	 * @return The uninstall {@link Artifact} for the install unit.
	 * If no such artifact exists, this method returns null.
	 */
	Artifact getUninstallArtifact();

	/**
	 * Gets the undo artifact.
	 * 
	 * @return The undo {@link Artifact} for the install unit.
	 * If no such artifact exists, this method returns null.
	 */
	Artifact getUndoArtifact();

	/**
	 * Gets the update artifact.
	 * 
	 * @return The update {@link Artifact} for the install unit.
	 * If no such artifact exists, this method returns null.
	 */
	Artifact getUpdateArtifact();
	
	/**
	 * Sets the target resource reference. 
	 * 
	 * The targetResourceRef attribute identifies the resource that will process the InstallableUnits artifacts.
     * The resources created or modified by artifact processing are frequently, but not necessarily, hosted by 
     * the target resource. 
     * 
     * This value MUST match an id of a resource element in Topology. The target may be a 
     * resource that has not yet been created. In this case, there is a dependency on the complete installation 
     * of the target resource prior to applying the InstallableUnit. This dependency MUST be represented in a 
     * Dependency element within Requirements that apply to the InstallableUnit.  
     *
	 * @param targetResourceReference Reference to the resource that can process the InstallableUnits artifacts.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setTargetResourceReference(String targetResourceReference)
		throws InvalidInputException;
	
	/**
     * Sets the languages.
     * 
     * When translated materials are deployed by the InstallableUnits artifacts, the languages of the 
     * translations are listed in Languages.  
     *   
     * @param languages A collection of languages supported by the InstallableUnit.
     */
	void setLanguages(Collection<Language> languages);
	
	/**
     * Sets the requirements.
     * 
     * Requirements specified in an InstallableUnit identify requirements that must be met prior to 
     * successful processing of the InstallableUnits artifacts. 
     *   
     * @param requirements A collection of requirements that must be met prior to successful processing of the 
     * InstallableUnits artifacts.
     */
	void setRequirements(Collection<Requirement> requirements);
	
	/**
     * Sets the required base.
     * 
     * When an InstallableUnit can be used to update resources, the RequiredBase element 
     * identifies the resources that can be updated. 
     *   
     * @param requiredBase A resource that will be updated when the InstallableUnits UpdateArtifact is processed.
     */
	void setRequiredBase(RequiredBase requiredBase);
	
	/**
     * Sets the resulting resources.
     * 
     * An InstallableUnits ResultingResource element identifies the resources in Topology that will be installed or 
     * updated when the InstallableUnits artifacts are processed.
     *   
     * @param resultingResources A collection of resources that will be installed or updated by processing 
     * the InstallableUnits artifacts.
     * @throws InvalidInputException if the parameter is null
     */
	void setResultingResources(Collection<ResultingResource> resultingResources);
	
	
	/**
	 * Sets the install artifact for the install unit.
	 * Setting to null removes the existing install artifact from
	 * the install unit, if it exists.
	 * 
	 * Note that install units are required to contain at least one artifact.
	 * Because this is true, attempts to remove the only artifact contained in
	 * an install unit result in an exception.
	 * 
	 * @param installArtifact The install Artifact for the install unit.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setInstallArtifact(Artifact installArtifact)
		throws InvalidInputException;

	/**
	 * Sets the uninstall artifact for the install unit.
	 * Setting to null removes the existing uninstall artifact from
	 * the install unit, if it exists.
	 * 
	 * Note that install units are required to contain at least one artifact.
	 * Because this is true, attempts to remove the only artifact contained in
	 * an install unit result in an exception.
	 * 
	 * @param uninstallArtifact The uninstall Artifact for the install unit.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setUninstallArtifact(Artifact uninstallArtifact)
		throws InvalidInputException;
	
	/**
	 * Sets the undo artifact for the install unit.
	 * Setting to null removes the existing undo artifact from
	 * the install unit, if it exists.
	 * 
	 * Note that install units are required to contain at least one artifact.
	 * Because this is true, attempts to remove the only artifact contained in
	 * an install unit result in an exception.
	 * 
	 * @param undoArtifact The undo Artifact for the install unit.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setUndoArtifact(Artifact undoArtifact)
		throws InvalidInputException;
	
	/**
	 * Sets the update artifact for the install unit.
	 * Setting to null removes the existing update artifact from
	 * the install unit, if it exists.
	 * 
	 * Note that install units are required to contain at least one artifact.
	 * Because this is true, attempts to remove the only artifact contained in
	 * an install unit result in an exception.
	 * 
	 * @param updateArtifact The update Artifact for the install unit.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setUpdateArtifact(Artifact updateArtifact)
		throws InvalidInputException;
	
}
