/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit;

import java.math.BigInteger;
import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * MultiSelectType defines a way to associate features with a defined minimum and maximum number 
 * of selections allowed. A MultiSelect element MAY be used to support identification of mutually 
 * exclusive features.
 */
public interface MultiFeatureSelect {
	/**
     * Gets the features.
     * 
     * The value of the feature MUST reference the id of a feature element.
     * 
     * @return A collection of references to features in the list of features defined in the MultiSelect element.
     */
    Collection<String> getFeatures();
    
    /**
     * Gets the minimum selections.
     * 
     * When it is not necessary that any of the features in the MultiSelect list be selected, the default of 0 can be used.
     * Mutually exclusive features can be defined using a MultiSelect element with two features, minSelections set to 0 and 
     * maxSelections set to 1. If multiple instances of a single feature are selected via multiplicity, the set of multiple 
     * instances count only once toward the minimum and maximum. In other words, the count is based solely on the features 
     * selected, not on how many instances of each feature are selected.
     * 
     * When maxSelections is not defined, all of the features in the MultiSelect MAY be selected for a particular deployment.
     * If defined, the maxSelections value MUST be greater than or equal to the minSelections value and MUST be less than or 
     * equal to the number of referenced features. 
     * 
     * @return Minimum number of features that must be selected.
     * **default value=0
     */
    BigInteger getMinSelections();
    
    /**
     * Gets the maximum selections.
     * 
     * When it is not necessary that any of the features in the MultiSelect list be selected, the default of 0 can be used.
     * Mutually exclusive features can be defined using a MultiSelect element with two features, minSelections set to 0 and 
     * maxSelections set to 1. If multiple instances of a single feature are selected via multiplicity, the set of multiple 
     * instances count only once toward the minimum and maximum. In other words, the count is based solely on the features 
     * selected, not on how many instances of each feature are selected.
     * 
     * When maxSelections is not defined, all of the features in the MultiSelect MAY be selected for a particular deployment.
     * If defined, the maxSelections value MUST be greater than or equal to the minSelections value and MUST be less than or 
     * equal to the number of referenced features. 
     * 
     * @return Maximum number of features that can be selected.
     * 
     */
    BigInteger getMaxSelections();
    
    /**
     * Sets the features.
     * 
     * The value of the feature MUST reference the id of a feature element.
     * 
     * @param features A collection of references to features in the list of features defined in the 
     * MultiSelect element.
     * @throws InvalidInputException if the parameter is null
     */
    void setFeatures(Collection<String> features);
    
    /**
     * Sets the minimum selections.
     * 
     * When it is not necessary that any of the features in the MultiSelect list be selected, the default of 0 can be used.
     * Mutually exclusive features can be defined using a MultiSelect element with two features, minSelections set to 0 and 
     * maxSelections set to 1. If multiple instances of a single feature are selected via multiplicity, the set of multiple 
     * instances count only once toward the minimum and maximum. In other words, the count is based solely on the features 
     * selected, not on how many instances of each feature are selected.
     * 
     * When maxSelections is not defined, all of the features in the MultiSelect MAY be selected for a particular deployment.
     * If defined, the maxSelections value MUST be greater than or equal to the minSelections value and MUST be less than or 
     * equal to the number of referenced features. 
     * 
     * @param numSelections Minimum number of features that must be selected.
     * **default value=0
     */
    void setMinSelections(BigInteger numSelections);
    
    /**
     * Gets the maximum selections.
     * 
     * When it is not necessary that any of the features in the MultiSelect list be selected, the default of 0 can be used.
     * Mutually exclusive features can be defined using a MultiSelect element with two features, minSelections set to 0 and 
     * maxSelections set to 1. If multiple instances of a single feature are selected via multiplicity, the set of multiple 
     * instances count only once toward the minimum and maximum. In other words, the count is based solely on the features 
     * selected, not on how many instances of each feature are selected.
     * 
     * When maxSelections is not defined, all of the features in the MultiSelect MAY be selected for a particular deployment.
     * If defined, the maxSelections value MUST be greater than or equal to the minSelections value and MUST be less than or 
     * equal to the number of referenced features. 
     * 
     * @param numSelections Maximum number of features that can be selected.
     * 
     */
    void setMaxSelections(BigInteger numSelections);
    
    /**
	 * Creates a copy of the object.
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();
}
