/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit;

import java.math.BigInteger;
import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * Some solutions allow multiple instances of some portion of the solutions resources to be deployed as part of the solution.
 * <ul> For example, a solution that includes a server and a client may allow the deployment of multiple clients. The deployment 
 * of each client may involve content elements that represent several different resulting resources, features that control optional 
 * functionality of the client and configuration elements that configure the client. All of these can be defined within a Client 
 * feature that declares a Multiplicity element that indicates that multiple clients are allowed. Each selection or instance of 
 * the feature results in the deployment of a client.</ul>
 * The phrase feature instance is used to refer to the set of instances of all resources deployed when the feature is selected. 
 * It does not imply that features themselves are represented as having lifecycle or that features in the SDD correspond with feature 
 * instances in the deployment environment.
 * 
 */
public interface Multiplicity {
	
    /**
     * Gets the constrained features.
     * 
     * A feature with multiplicity may contain NestedFeature elements. When a NestedFeature is identified in a ConstrainedFeature, 
     * then all instances of the defining Feature MUST make the same selection choice for that NestedFeature.
     * 
     * @return A collection of nested features whose selection must be the same for all instances of the defining feature in a particular deployment.
     */
    Collection<String> getConstrainedFeatures();
    
    /**
     * Gets the constrained resources.
     * 
     * The content elements selected by a feature may express constraints on resources. When the resource constraints for each 
     * instance of a feature must resolve to the same resource instance, or when all must resolve to unique resource instances, 
     * the resource is referred to and the constraint type is identified in the ConstrainedResource element.
     *  
     * @return A collection of resources that must resolve to the same resource instance for all instances of the feature in a particular deployment.
     */
    Collection<ConstrainedResource> getConstrainedResources();
    
    /**
     * Gets whether multiples are allowed.
     * 
     * This is an attribute with a fixed value of true. It is included because all other elements and attributes of 
     * MultiplicityType are optional. A feature that allows multiples but has no need to define constraints on resources, 
     * features or number of instances would define a Multiplicity element that had only the multiplesAllowed attribute.
     *  
     * @return Indicates that multiple instances of the feature are allowed.
     */
    Boolean getMultiplesAllowed();
    
    /**
     * Gets the minimum selection allowed.
     * 
     * When a feature is selected, if more than one instance of the feature is required, minSelections MUST be specified. 
     * 
     * @return The minimum number of instances of the feature that must be selected if the feature is selected at all.
     * 
     */
    BigInteger getMinSelections();
    
    /**
     * Gets the maximum selection allowed.
     * 
     * When a feature is selected, if there is a limit on the number of instances of the feature that can be selected, 
     * maxSelections MUST be specified. If maxSelections is defined, it MUST be equal to or greater than minSelections.
     * 
     * @return The maximum number of instances of the feature that can be selected.
     */
    BigInteger getMaxSelections();

    /**
     * Sets the constrained features.
     * 
     * A feature with multiplicity may contain NestedFeature elements. When a NestedFeature is identified in a ConstrainedFeature, 
     * then all instances of the defining Feature MUST make the same selection choice for that NestedFeature.
     * 
     * @param constrainedFeatureSelections A collection of nested features whose selection must be the same for all instances of the defining 
     * feature in a particular deployment.
     */
    void setConstrainedFeatures(Collection<String> constrainedFeatureSelections);
    
    /**
     * Sets the constrained resources.
     * 
     * The content elements selected by a feature may express constraints on resources. When the resource constraints for each 
     * instance of a feature must resolve to the same resource instance, or when all must resolve to unique resource instances, 
     * the resource is referred to and the constraint type is identified in the ConstrainedResource element.
     * 
     * @param constrainedResourceInstances A collection of resources that must resolve to the same resource instance for all instances of the 
     * feature in a particular deployment.
     */
    void setConstrainedResources(Collection<ConstrainedResource> constrainedResourceInstances);
    
    /**
     * Sets whether multiples allowed.
     * 
     * This is an attribute with a fixed value of true. It is included because all other elements and attributes of 
     * MultiplicityType are optional. A feature that allows multiples but has no need to define constraints on resources, 
     * features or number of instances would define a Multiplicity element that had only the multiplesAllowed attribute.
     * 
     * @param multiplesAllowed Indicates that multiple instances of the feature are allowed.
     * <ul>**fixed value=true</ul>
     * @throws InvalidInputException if the parameter is null
     */
    void setMultiplesAllowed(Boolean multiplesAllowed);
    
    /**
     * Sets the minimum selection allowed.
     * 
     * When a feature is selected, if more than one instance of the feature is required, minSelections MUST be specified. 
     * 
     * @param minSelections The minimum number of instances of the feature that must be selected if the feature is selected at all.
     * <ul>**default value=1</ul>
     */
    void setMinSelections(BigInteger minSelections);
    
    /**
     * Sets the maxmum selection allowed.
     * 
     * When a feature is selected, if there is a limit on the number of instances of the feature that can be selected, 
     * maxSelections MUST be specified. If maxSelections is defined, it MUST be equal to or greater than minSelections.
     * 
     * @param maxSelections The maximum number of instances of the feature that can be selected.
     */
    void setMaxSelections(BigInteger maxSelections);
    
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
    public Object clone();
}
