/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit;

import java.util.Collection;
import java.util.Properties;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Describable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * InstallableUnit and LocalizationUnit content elements can include zero or more ResultingResource elements that describe the 
 * key resources installed or updated when the content elements artifacts are processed. The type definition for these elements 
 * is provided by ResultingResourceType. ResultingResource elements refer to resources in topology and define characteristics of 
 * those resources that will become true when the artifact is applied. The deployment descriptor author MAY omit the 
 * ResultingResource element from the content element and the definition of the resource from Topology when no knowledge of 
 * their existence is required for deployment of the solution or for aggregation of the solution. Characteristics that exist in 
 * ResultingResource and elsewhere, such as Topology or ResultingChange, MUST NOT conflict.
 * <ul>For example, if Topology specifies a property that indicates that a file must be writable, it would be incorrect for 
 * ResultingResource to specify that the resulting file resource is read-only.</ul>
 * Example uses of the ResultingResource element are to:
 * <li>determine whether potentially resulting resources will actually be installed or updated;</li>
 * <li>identify the resource associated with a content element that may be subsequently uninstalled using the uninstall information 
 * in this SDD;</li>
 * <li>discover the components of a logical solution resource previously installed using this SDD;</li>
 * <li>check whether or not a content element has already been installed.</li>
 *
 */
public interface ResultingResource extends Describable {
	
    /**
     * Gets the name.
     * 
     * The name of the resulting resource SHOULD be defined in the ResultingResource element and not in Topology when the 
     * content element installs the resulting resource. The resource name comes into existence when the resulting resource 
     * is created. When the content element updates the resulting resource without changing the resource name, Name SHOULD 
     * be defined in Topology. Name SHOULD NOT be defined in both places. If a resource name is defined in both Topology 
     * and ResultingResource, the values MUST match.
     * 
     * @return Name of the resulting resource as known in the deployment environment.
     */
    String getName();
    
	/**
	 * Gets the resource reference.
	 * 
	 * The resourceRef attribute MUST identify the resource in Topology that will be installed or updated when the defining 
	 * content element is applied.
	 * 
	 * @return Reference to a resource in topology.
	 */
	String getResourceReference();
	
	/**
	 * Gets the version.
	 * 
	 * This is the version of the resource after processing the content elements artifacts. Version SHOULD be defined for 
	 * all resulting resources.
	 * <ul>For example, when update artifacts are processed, this version describes the resource after the update is complete.</ul>
	 * 
	 * @return Version of the resulting resource.
	 */
	String getVersion();

	/**
	 * Gets the fix names.
	 * 
	 * Multiple FixName elements MAY be included to identify the resulting resource fixes that will exist once the content element 
	 * is applied. The FixName SHOULD match the names of fixes that can be detected on the system.
	 * 
	 * @return Names of resulting fixes.
	 */
	Collection<String> getFixNames();

	/**
	 * Gets the properties.
	 * 
	 * Property elements SHOULD be included to identify property values of the resulting resource that will exist after applying 
	 * the content element.<br><br>
	 * Properties of the resulting resource SHOULD be defined in the ResultingResource element and not in Topology. They SHOULD NOT 
	 * be defined in both places. If a property is defined in both Topology and ResultingResource, the values MUST match.
	 * 
	 * @return A resulting property setting of the resulting resource.
	 */
	Properties getProperties();
	
	/**
	 * Gets the condition.
	 * 
	 * A Condition is used when the resulting resource will be created by the content element only when certain conditions exist 
	 * in the deployment environment.
	 * 
	 * @return A condition that determines if the resulting resource definition is relevant to a particular deployment.
	 */
	Condition getCondition();
	
	/**
	 * Sets the name.
	 * 
	 * The name of the resulting resource SHOULD be defined in the ResultingResource element and not in Topology when the 
     * content element installs the resulting resource. The resource name comes into existence when the resulting resource 
     * is created. When the content element updates the resulting resource without changing the resource name, Name SHOULD 
     * be defined in Topology. Name SHOULD NOT be defined in both places. If a resource name is defined in both Topology 
     * and ResultingResource, the values MUST match.
     * 
	 * @param name Name of the resulting resource as known in the deployment environment.
	 */
	void setName(String name);
	
	/**
	 * Sets the resource reference.
	 * 
	 * The resourceRef attribute MUST identify the resource in Topology that will be installed or updated when the defining 
	 * content element is applied.
	 * 
	 * @param resourceReference Reference to a resource in topology.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setResourceReference(String resourceReference);
	
	/**
	 * Sets the version.
	 * 
	 * This is the version of the resource after processing the content elements artifacts. Version SHOULD be defined for 
	 * all resulting resources.
	 * <ul>For example, when update artifacts are processed, this version describes the resource after the update is complete.</ul>
	 * 
	 * @param version Version of the resulting resource.
	 */
	void setVersion(String version);
	
	/**
	 * Sets the fix names.
	 * 
	 * Multiple FixName elements MAY be included to identify the resulting resource fixes that will exist once the content element 
	 * is applied. The FixName SHOULD match the names of fixes that can be detected on the system.
	 * 
	 * @param fixNames Names of resulting fixes.
	 */
	void setFixNames(Collection<String> fixNames);
	
	/**
	 * Sets the properties.
	 * 
	 * Property elements SHOULD be included to identify property values of the resulting resource that will exist after applying 
	 * the content element.<br><br>
	 * Properties of the resulting resource SHOULD be defined in the ResultingResource element and not in Topology. They SHOULD NOT 
	 * be defined in both places. If a property is defined in both Topology and ResultingResource, the values MUST match.
	 * 
	 * @param resultingResourceProperties A resulting property setting of the resulting resource.
	 */
	void setProperties(Properties resultingResourceProperties);
	
	/**
	 * Sets the condition.
	 * 
	 * A Condition is used when the resulting resource will be created by the content element only when certain conditions exist 
	 * in the deployment environment.
	 *  
	 * @param condition A condition that determines if the resulting resource definition is relevant to a particular deployment.
	 */
	void setCondition(Condition condition);

	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	Object clone();
}
