/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement;

import java.math.BigInteger;
import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.DisplayNameable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * AlternativeRequirementType provides the type definition for Alternative elements used within requirements to define alternative 
 * sets of resource constraints that will satisfy the requirement.
 */
public interface AlternativeRequirement extends DisplayNameable {
	/**
	 * Gets the alternative requirement ID.
	 * 
	 * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
	 * 
	 * @return Identifier for the alternative scoped to the deployment descriptor.
	 */
	String getAlternativeRequirementID();
	
	/**
	 * Gets the priority.
	 * 
	 * If there are multiple satisfied alternatives during a particular solution deployment, one of the alternatives must be selected. 
	 * The priority attribute communicates the SDD authors prioritization of the alternatives. A lower number represents a higher 
	 * priority with 1 representing the highest priority. Other inputs may also be used to select an alternative. The criteria for 
	 * making this selection are outside of the scope of the SDD.
	 * 
	 * @return Assists in determining alternative selected when multiple alternatives evaluate to true.
	 */
	BigInteger getPriority();
	
	/**
	 * Gets the resource constraints.
	 * 
	 * Every ResourceConstraint defined in a single Alternative MUST be met for the alternative requirement to be considered satisfied. 
	 * 
	 * @return A collection of requirements on one resource.
	 */
	Collection<RequirementResourceConstraint> getResourceConstraints();
	 
	/**
	 * Sets the alternative requirement ID.
	 * 
	 * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
	 * 
	 * @param alternativeRequirementID Identifier for the alternative scoped to the deployment descriptor.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setAlternativeRequirementID(String alternativeRequirementID) throws InvalidInputException;
	
	/**
	 * Sets the priority.
	 * 
	 * If there are multiple satisfied alternatives during a particular solution deployment, one of the alternatives must be selected. 
	 * The priority attribute communicates the SDD authors prioritization of the alternatives. A lower number represents a higher 
	 * priority with 1 representing the highest priority. Other inputs may also be used to select an alternative. The criteria for 
	 * making this selection are outside of the scope of the SDD.
	 * 
	 * @param priority Assists in determining alternative selected when multiple alternatives evaluate to true.
	 * <ul>**default value=1</ul>
	 * 
	 */
	void setPriority(BigInteger priority);
	
	/**
	 * Sets the resource constraints.
	 * 
	 * Every ResourceConstraint defined in a single Alternative MUST be met for the alternative requirement to be considered satisfied. 
	 * 
	 * @param resourceConstraints A collection of requirements on one resource.
	 */
	void setResourceConstraints(Collection<RequirementResourceConstraint> resourceConstraints);
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	Object clone();
}