/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Describable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * CapacityConstraintType provides the type definition of the Capacity elements of RequirementResourceConstraintType. These elements 
 * are used to express a requirement on the capacity of a particular resource property such as memory available from an operating 
 * system. Capacity is shared: multiple capacity constraints expressed on the same property are evaluated individually without assuming 
 * any change to the available quantity of the property. 
 */
public interface CapacityConstraint extends Describable {
	/**
	 * Gets the property name.
	 * 
	 * This name corresponds to the name of the constrained resource property in the environment. This name may be specified in 
	 * profiles.
	 * 
	 * @return Name of the constrained property.
	 */
	String getPropertyName();
	
	/**
	 * Gets the unit of measure.
	 * 
	 * Values for unit SHOULD be well-known units of measure from the International System of Units [UNIT]. A unit of measure SHOULD 
	 * be specified for all properties that are measured in any kind of unit.
	 * 
	 * @return Unit of measure used to interpret the capacity value.
	 */
	String getUnitOfMeasure();
	
	/**
	 * Gets the minimum capacity.
	 * 
	 * There will usually be either a minimum value or a maximum value defined, but not both. When minimum is specified, the actual 
	 * value of the capacity property MUST be equal to or greater than the minimum value.
	 * 
	 * @return Minimum capacity.
	 */
	String getMinimum();
	
	/**
	 * Gets the maximum capacity.
	 * 
	 * When specified, the actual value of the capacity property MUST be less than or equal to the defined maximum.<br><br>
	 * If Minimum and Maximum are both defined, Minimum MUST be less than or equal to Maximum.
	 * 
	 * @return Maximum capacity.
	 */
	String getMaximum();
	
	/**
	 * Gets the recommended minimum capacity.
	 * 
	 * The SDD author can indicate a preferred, but not required, minimum by defining a value for this element.
	 * 
	 * @return Minimum recommended capacity.
	 */
	String getRecommendedMinimum();
	
	/**
	 * Gets the recommanded maximum capacity.
	 * 
	 * The SDD author can indicate a preferred, but not required, maximum by defining a value for this element.<br><br>
	 * If MinimumRecommended and MaximumRecommended are both defined, MinimumRecommended MUST be less than or equal to 
	 * MaximumRecommended.
	 * 
	 * @return Maximum recommended capacity.
	 */
	String getRecommendedMaximum();
	
	
	/**
	 * Sets the property name.
	 * 
	 * This name corresponds to the name of the constrained resource property in the environment. This name may be specified in 
	 * profiles.
	 * 
	 * @param propertyName Name of the constrained property.
	 */
	void setPropertyName(String propertyName);
	
	/**
	 * Sets the unit of measure.
	 * 
	 * Values for unit SHOULD be well-known units of measure from the International System of Units [UNIT]. A unit of measure SHOULD 
	 * be specified for all properties that are measured in any kind of unit.
	 * 
	 * @param unitOfMeasure Unit of measure used to interpret the capacity value.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setUnitOfMeasure(String unitOfMeasure);
	
	/**
	 * Sets the minimum capacity.
	 * 
	 * There will usually be either a minimum value or a maximum value defined, but not both. When minimum is specified, the actual 
	 * value of the capacity property MUST be equal to or greater than the minimum value.
	 * 
	 * @param minimum Minimum capacity.
	 */
	void setMinimum(String minimum);
	
	/**
	 * Sets the maximum capacity.
	 * 
	 * When specified, the actual value of the capacity property MUST be less than or equal to the defined maximum.<br><br>
	 * If Minimum and Maximum are both defined, Minimum MUST be less than or equal to Maximum.
	 * 
	 * @param maximum Maximum capacity.
	 */
	void setMaximum(String maximum);
	
	/**
	 * Sets the recommended minimum capacity.
	 * 
	 * The SDD author can indicate a preferred, but not required, minimum by defining a value for this element.
	 * 
	 * @param recommendedMinimum Minimum recommended capacity.
	 */
	void setRecommendedMinimum(String recommendedMinimum);
	
	/**
	 * Sets the recommended capacity.
	 * The SDD author can indicate a preferred, but not required, maximum by defining a value for this element.<br><br>
	 * If MinimumRecommended and MaximumRecommended are both defined, MinimumRecommended MUST be less than or equal to 
	 * MaximumRecommended.
	 * 
	 * @param recommendedMaximum Maximum recommended capacity.
	 */
	void setRecommendedMaximum(String recommendedMaximum);
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();
}