/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.DisplayNameable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * ResourceConstraintType provides the Type section for the ResourceConstraint element in content element Requirements. A 
 * ResourceConstraint is a set of zero or more constraints on one resource. 
 */
public interface RequirementResourceConstraint extends DisplayNameable { 
    /**
     * Gets the name.
     * 
     * This name is used to identify the resource in the deployment environment. If the resource identified by resourceRef does not 
     * have the name defined here, then the constraint is not met.
     * 
     * @return The name of the resource.
     */
    String getName();
    
	/**
	 * Gets the resource constraint ID.
	 * 
	 * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
	 * 
	 * @return Identifier for the ResourceConstraint scoped to the deployment descriptor.
	 */
	String getResourceConstraintID();
	
	/**
	 * Gets the resource reference.
	 * 
	 * This is the resource to which the constraints apply.<br><br>
	 * This reference MUST refer to the id of a resource in Topology.
	 * 
	 * @return Reference to a resource specification in topology.
	 */
	String getResourceReference();
	
	/**
	 * Gets whether the ResourceConstraint must evaluate to true or to false.
	 * 
	 * When the result of evaluating Name and all of the constraints defined in the ResourceConstraint matches the value of testValue, 
	 * the ResourceConstraint is considered met.<br><br>
	 * When no Name or constraints are defined, and testValue is true, the constraint is met if the resource exists as defined in 
	 * topology.<br><br>
	 * When no Name or constraints are defined, and testValue is false, the constraint is met if the resource, as defined in topology, 
	 * does not exist.
	 * 
	 * @return Indicates whether the ResourceConstraint must evaluate to true or to false.
	 */
	Boolean getTestValue();
	
	/**
	 * Gets the capacity constraints.
	 * 
	 * A bound on a quantifiable property of a resource.
	 * 
	 * @return Capacity constraints that apply to the resource identified in resourceRef.
	 */
	Collection<CapacityConstraint> getCapacityConstraintArray();

	/**
	 * Gets the consumption constraints.
	 * 
	 * A required quantity of a property of a resource in any state.
	 * 
	 * @return Consumption constraints that apply to the resource identified in resourceRef.
	 */
	Collection<ConsumptionConstraint> getConsumptionConstraintArray();
	
	/**
	 * Gets the property constraints.
	 * 
	 * A required value or set of values of a property.
	 * 
	 * @return Property constraints that apply to the resource identified in resourceRef.
	 */
	Collection<PropertyConstraint> getPropertyConstraintArray();
    
	/**
	 * Gets the version constraints.
	 * 
	 * A required value or set of values of a version property.
	 * 
	 * @return Version constraints that apply to the resource identified in resourceRef.
	 */
	Collection<VersionConstraint> getVersionConstraintArray();
	
	/**
	 * Gets the uniqueness constraints.
	 * 
	 * A required mapping of two resources in the topology to unique instances in the deployment environment.
	 * 
	 * @return Uniqueness constraints that represent required mappings of two resources in the topology to unique instances in the deployment environment.
	 */
	Collection<UniquenessConstraint> getUniquenessConstraintArray();
	
	/**
	 * Gets the relationship constraints.
	 * 
	 * A required relationship between the resource identified in the resourceRef and another resource in the topology.
	 * 
	 * @return Relationship constraints that represent required relationships between the resource identified in the resourceRef and another resource in the topology.
	 */
	Collection<RelationshipConstraint> getRelationshipConstraintArray();
	
	/**
	 * Sets the name.
	 * 
	 * This name is used to identify the resource in the deployment environment. If the resource identified by resourceRef does not 
     * have the name defined here, then the constraint is not met.
     * 
	 * @param name The name of the resource.
	 */
	void setName(String name);
	
	/**
	 * Sets the resource constraint ID.
	 * 
	 * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
	 * 
	 * @param resourceConstraintID Identifier for the ResourceConstraint scoped to the deployment descriptor.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setResourceConstraintID(String resourceConstraintID);
	
	/**
	 * Sets the resource reference.
	 * 
	 * This is the resource to which the constraints apply.<br><br>
	 * This reference MUST refer to the id of a resource in Topology.
	 * 
	 * @param resourceReference Reference to a resource specification in topology.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setResourceReference(String resourceReference);
	
	/**
	 * Sets whether the ResourceConstraint must evaluate to true or to false.
	 * 
	 * When the result of evaluating Name and all of the constraints defined in the ResourceConstraint matches the value of testValue, 
	 * the ResourceConstraint is considered met.<br><br>
	 * When no Name or constraints are defined, and testValue is true, the constraint is met if the resource exists as defined in 
	 * topology.<br><br>
	 * When no Name or constraints are defined, and testValue is false, the constraint is met if the resource, as defined in topology, 
	 * does not exist.
	 * 
	 * @param testValue Indicates whether the ResourceConstraint must evaluate to true or to false.
	 * <ul>**default value=true.</ul>
	 * 
	 */
	void setTestValue(Boolean testValue);
	
	/**
	 * Sets the capacity constraints.
	 * 
	 * A bound on a quantifiable property of a resource.
	 * 
	 * @param capacityConstraintArray Capacity constraints that apply to the resource identified in resourceRef.
	 */
	void setCapacityConstraintArray(Collection<CapacityConstraint> capacityConstraintArray);

	/**
	 * Sets the consumption constraints.
	 * 
	 * A required quantity of a property of a resource in any state.
	 * 
	 * @param consumptionConstraintArray Consumption constraints that apply to the resource identified in resourceRef.
	 */
	void setConsumptionConstraintArray(Collection<ConsumptionConstraint> consumptionConstraintArray);
	
    /**
     * Sets the property constraints.
     * 
     * A required value or set of values of a property.
     * 
     * @param propertyConstraintArray Property constraints that apply to the resource identified in resourceRef.
     */
    void setPropertyConstraintArray(Collection<PropertyConstraint> propertyConstraintArray);
    
	/**
	 * Sets the version constraints.
	 * 
	 * A required value or set of values of a version property.
	 * 
	 * @param versionConstraintArray Version constraints that apply to the resource identified in resourceRef.
	 */
	void setVersionConstraintArray(Collection<VersionConstraint> versionConstraintArray);
	
	/**
	 * Sets the uniqueness constraints.
	 * 
	 * A required mapping of two resources in the topology to unique instances in the deployment environment.
	 * 
	 * @param uniquenessConstraintArray Uniqueness constraints that represent required mappings of two resources in the topology to unique instances in the deployment environment.
	 */
	void setUniquenessConstraintArray(Collection<UniquenessConstraint> uniquenessConstraintArray);
	
	/**
	 * Sets the relationship constraints.
	 * 
	 * A required relationship between the resource identified in the resourceRef and another resource in the topology.
	 * 
	 * @param relationshipConstraintArray Relationship constraints that represent required relationships between the resource identified in the resourceRef and another resource in the topology.
	 */
	void setRelationshipConstraintArray(Collection<RelationshipConstraint> relationshipConstraintArray);

	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	Object clone();
}