/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Describable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.VersionRange;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.VersionValue;

/**
 * VersionConstraintType provides the type definition of the VersionConstraint elements of RequirementResourceConstraintType. A 
 * VersionConstraint can define a set of individual versions or ranges of versions that are supported and a similar set that are 
 * certified.
 */
public interface VersionConstraint extends Describable {
    /**
     * Gets the supported version values.
     * 
     * If the resource version is in the Supported set, it meets the requirements. 
     * 
     * Discrete version values can be defined when the set of required versions includes versions that do not fall within a range. 
     * There is no assumption by this specification that version values are numerically comparable. The method of comparing version 
     * values may be resource-specific. 
     * 
     * @return A collection of supported versions. Version values with associated fixes specified.
     */
    Collection<VersionValue> getVersionValues();
	
    /**
     * Gets the supported version ranges.
     * 
     * If the resource version is in the Supported set, it meets the requirements. 
     * 
     * A VersionRange is specified with a minimum and maximum version value and a list of required fixes associated with that range. 
     * The method of comparing version strings in a version range is resource-specific.
     * 
     * @return A collection of supported versions. Range of version values with associated fixes specified for each range.
     */
    Collection<VersionRange> getVersionRanges();
	
    /**
     * Gets the certified version ranges.
     * 
     * In some cases the set of required versions may be different from the set of versions that are certified by the manufacturer as thoroughly tested. 
     * 
     * A VersionRange is specified with a minimum and maximum version value and a list of required fixes associated with that range. 
     * The method of comparing version strings in a version range is resource-specific.
     * 
     * @return A subset of the supported versions that are certified as tested. Range of version values with associated fixes specified for each range.
     */
    Collection<VersionRange> getCertifiedVersionRanges();
	
    /**
     * Gets the certified version values.
     * 
     * In some cases the set of required versions may be different from the set of versions that are certified by the manufacturer as thoroughly tested. 
     * 
     * Discrete version values can be defined when the set of required versions includes versions that do not fall within a range. 
     * There is no assumption by this specification that version values are numerically comparable. The method of comparing version 
     * values may be resource-specific. 
     * 
     * @return A subset of the supported versions that are certified as tested. Version values with associated fixes specified.
     */
    Collection<VersionValue> getCertifiedVersionValues();
	
	/**
	 * Sets the supported version values.
	 * 
	 * If the resource version is in the Supported set, it meets the requirements. 
     * 
	 * Discrete version values can be defined when the set of required versions includes versions that do not fall within a range. 
     * There is no assumption by this specification that version values are numerically comparable. The method of comparing version 
     * values may be resource-specific. 
     * 
	 * @param versionValues A collection of supported versions. Version values with associated fixes specified.
	 */
	void setVersionValues(Collection<VersionValue> versionValues);
	
	/**
	 * Sets the supported version ranges.
     * 
     * If the resource version is in the Supported set, it meets the requirements. 
     * 
	 * A VersionRange is specified with a minimum and maximum version value and a list of required fixes associated with that range. 
     * The method of comparing version strings in a version range is resource-specific.
     * 
	 * @param versionRanges A collection of supported versions. Ranges of version values with associated fixes specified for each range.
	 */
	void setVersionRanges(Collection<VersionRange> versionRanges);
	
	/**
	 * Sets the certified version values.
     * 
     * In some cases the set of required versions may be different from the set of versions that are certified by the manufacturer as thoroughly tested. 
     * 
	 * Discrete version values can be defined when the set of required versions includes versions that do not fall within a range. 
     * There is no assumption by this specification that version values are numerically comparable. The method of comparing version 
     * values may be resource-specific. 
     * 
	 * @param certifiedVersionValues A subset of the supported versions that are certified as tested. Version values with associated fixes specified.
	 */
	void setCertifiedVersionValues(Collection<VersionValue> certifiedVersionValues);
	
	/**
	 * Sets the certified version ranges.
     * 
     * In some cases the set of required versions may be different from the set of versions that are certified by the manufacturer as thoroughly tested. 
     * 
     * A VersionRange is specified with a minimum and maximum version value and a list of required fixes associated with that range. 
     * The method of comparing version strings in a version range is resource-specific.
     * 
	 * @param certifiedVersionRanges A subset of the supported versions that are certified as tested. Ranges of version values with associated fixes specified for each range.
	 */
	void setCertifiedVersionRanges(Collection<VersionRange> certifiedVersionRanges);
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	Object clone();
}