/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable;

import java.math.BigInteger;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * ConditionalDerivedVariableExpressionType is the type of the ConditionalExpression elements in derived variables. These 
 * elements associate a condition with a variable expression.
 */
public interface ConditionalDerivedVariableExpression {
	/**
	 * Gets the condition.
	 * 
	 * Selection of conditioned expressions is based on the characteristics of one or more resources that participate in a particular 
	 * solution deployment. These characteristics are defined in the Condition element.
	 * 
	 * @return A set of resource characteristics that are evaluated to determine if the associated expression is a candidate 
	 * for determining the value of the derived variable.
	 */
	Condition getCondition();
	
	/**
	 * Gets the expression.
	 * 
	 * The Expression element contains the expressions that evaluate to a potential value of the DerivedVariable. Only one 
	 * expression will be selected for use in a particular solution deployment.
	 * 
	 * @return Evaluation of this expression produces a candidate value for the derived variable.
	 */
	String getExpression();
	
	/**
	 * Gets the priority.
	 * 
	 * When multiple conditions evaluate to true for a particular deployment, the expression chosen is determined by the 
	 * priority value. A higher priority is indicated by a lower value. 1 is the highest priority.
	 *  
	 * @return A priority used as a tie-breaker when multiple expressions are available to determine the value of the variable.
	 */
	BigInteger getPriority();
		
	/**
	 * Sets the condition.
	 * 
	 * Selection of conditioned expressions is based on the characteristics of one or more resources that participate in a particular 
	 * solution deployment. These characteristics are defined in the Condition element.
	 * 
	 * @param newCondition A set of resource characteristics that are evaluated to determine if the associated expression 
	 * is a candidate for determining the value of the derived variable.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setCondition(Condition newCondition);
	
	/**
	 * Sets the expression.
	 * 
	 * The Expression element contains the expressions that evaluate to a potential value of the DerivedVariable. Only one 
	 * expression will be selected for use in a particular solution deployment.
	 * 
	 * @param expression Evaluation of this expression produces a candidate value for the derived variable.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setExpression(String expression);
	
	/**
	 * Sets the priority.
	 * 
	 * When multiple conditions evaluate to true for a particular deployment, the expression chosen is determined by the 
	 * priority value. A higher priority is indicated by a lower value. 1 is the highest priority.
	 *  
	 * @param priority A priority used as a tie-breaker when multiple expressions are available to determine the value of the variable.
	 * <ul>**default value=1</ul>
	 */
	void setPriority(BigInteger priority);
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	Object clone();
}