/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.DisplayNameable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * BaseParameterType provides a default value, along with other attributes used by all parameter types. It also provides the id 
 * attribute, which is used to reference the parameter in variable expressions.
 */
public interface Parameter extends DisplayNameable {
    /**
     * Gets the default value.
     * 
     * The defaultValue is used if no other value is provided as input to the deployment process.<br><br>
     * The value is interpreted based on the type of the defining parameter. 
     * <ul>For example, the defaultValue for a BooleanParameter must be either true or false; the defaultValue for a 
     * StringParameter must be a string; etc.</ul>
     * 
     * @return Default value for the parameter.
     */
    String getDefaultValue();
    
    /**
     * Gets the operations.
     * 
     * This attribute enables unique parameters to be defined per operation. Note that the use of a parameter for a particular 
     * operation is determined by a reference to the parameter in a variable expression or artifact argument used when performing 
     * that operation. The operation(s) associated with a parameters use can be determined by examining its use in the SDD. The 
     * operation attribute provides a quick way to know which operation(s) will use the parameter without having to examine the use 
     * of the parameter.<br><br>
     * All parameters defined within a CompositeInstallable are associated with the single operation supported by the 
     * CompositeInstallalbe. The operation attribute SHOULD NOT be set in this situation. 
     * 
     * The parameter is used when the specified operation(s) is (are) performed.
     * 
     * @return A collection of operations.
     */
    Collection<String> getOperations();
    
    /**
     * Gets the parameter ID.
     * 
     * Parameters may be referenced in DeploymentDescriptor elements of type VariableExpression within the scope of the parameter 
     * variable. The scope of the variable includes the content element where the variable is defined and all nested content elements. 
     * Variables defined in the top level content element are also visible in Topology. The Variable is referenced by placing the 
     * variable id within parentheses preceded by a dollar sign. 
     * <ul>For example, a variable with id value InstallLocation is referenced with the string $(InstallLocation).</ul>
     * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
     * 
     * @return Identifier used for referencing the variable within the descriptor.
     */
    String getParameterID();
    
    /**
     * Gets whether a value for the parameter must be provided.
     * 
     * A true value for required indicates that a value for the parameter must be provided when the parameter is in scope for 
     * a particular deployment.<br><br>
     * In cases where the parameter should be ignored when the value expression is not valid for a particular deployment, set required 
     * to "false".<br><br>
     * A false value for the required attribute has no effect when defaultValue is set.
     * 
     * @return Whether a value for the parameter must be provided. A true value indicates that a value for the parameter must be provided.
     */
    Boolean isRequired();
    
    /**
     * Gets whether the variable contains sensitive data.
     * 
     * The sensitive attribute provides an indication of whether the data within a variable is likely to be considered sensitive. 
     * User name and password are examples of data that may be considered sensitive.
     * <ul>For example, sensitive data typically would not be displayed in a user interface, written to a log file, stored without 
     * protection, or in any way made visible except to authorized users.</ul>
     *  
     * @return Whether the variable contains sensitive data. A true value indicates the variable contains sensitive data.
     */
    Boolean isSensitive();
    
    /**
     * Sets the operations.
     * 
     * This attribute enables unique parameters to be defined per operation. Note that the use of a parameter for a particular 
     * operation is determined by a reference to the parameter in a variable expression or artifact argument used when performing 
     * that operation. The operation(s) associated with a parameters use can be determined by examining its use in the SDD. The 
     * operation attribute provides a quick way to know which operation(s) will use the parameter without having to examine the use 
     * of the parameter.<br><br>
     * All parameters defined within a CompositeInstallable are associated with the single operation supported by the 
     * CompositeInstallalbe. The operation attribute SHOULD NOT be set in this situation. 
     * 
     * The parameter is used when the specified operation(s) is (are) performed.
     * 
     * @param operations A collection of operations.
     */
    void setOperations(Collection<String> operations);

    /**
     * Sets the parameter ID.
     * 
     * Parameters may be referenced in DeploymentDescriptor elements of type VariableExpression within the scope of the parameter 
     * variable. The scope of the variable includes the content element where the variable is defined and all nested content elements. 
     * Variables defined in the top level content element are also visible in Topology. The Variable is referenced by placing the 
     * variable id within parentheses preceded by a dollar sign. 
     * <ul>For example, a variable with id value InstallLocation is referenced with the string $(InstallLocation).</ul>
     * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
     * 
     * @param parameterID Identifier used for referencing the variable within the descriptor.
     */
    void setParameterID(String parameterID);
    
    /**
     * Sets whether a value for the parameter must be provided.
     *  
     * A true value for required indicates that a value for the parameter must be provided when the parameter is in scope for 
     * a particular deployment.<br><br>
     * In cases where the parameter should be ignored when the value expression is not valid for a particular deployment, set required 
     * to "false".<br><br>
     * A false value for the required attribute has no effect when defaultValue is set.
     * 
     * @param required Whether a value for the parameter must be provided. A true value indicates that a value for the parameter must be provided.
     * <ul>**default value=true</ul>
     * @throws InvalidInputException if the parameter is null
     */
    void setRequired(Boolean required);
    
    /**
     * Sets whether the variable contains sensitive data.
     * 
     * The sensitive attribute provides an indication of whether the data within a variable is likely to be considered sensitive. 
     * User name and password are examples of data that may be considered sensitive.
     * <ul>For example, sensitive data typically would not be displayed in a user interface, written to a log file, stored without 
     * protection, or in any way made visible except to authorized users.</ul>
     *
     * @param sensitive Whether the variable contains sensitive data. A true value indicates the variable contains sensitive data.
     * <ul>**default value=false</ul>
     */
    void setSensitive(Boolean sensitive);

    /**
     * Sets the default value.
     * 
     * The defaultValue is used if no other value is provided as input to the deployment process.<br><br>
     * The value is interpreted based on the type of the defining parameter. 
     * <ul>For example, the defaultValue for a BooleanParameter must be either true or false; the defaultValue for a 
     * StringParameter must be a string; etc.</ul>
     * 
     * @param defaultValue Default value for the parameter.
     */
    void setDefaultValue(String defaultValue);
}
