/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable;

import java.util.Collection;

/**
 * Parameters are variables whose value is expected to be received as input to the deployment process. The SDD author can 
 * specify multiple specific types of parameters, including validation rules for the values of the parameters.
 */
public interface ParameterGroup {
    /**
     * Gets the boolean parameters.
     * 
     * BooleanParameterType extends BaseParameterType without adding any additional attributes or elements. When the defaultValue 
     * attribute is defined for a boolean parameter, its value MUST be either true or false.
     * See {@link BooleanParameter}
     * 
     * @return Boolean input parameters.
     */
    Collection<BooleanParameter> getBooleanParameters();
	
    /**
     * Gets the integer parameters.
     * 
     * IntegerParameterType defines upper and lower bounds that can be used to validate the input received for that parameter.
     * See {@link IntegerParameter}
     * 
     * @return Integer input parameters.
     */
    Collection<IntegerParameter> getIntegerParameters();
	
    /**
     * Gets the string parameters.
     * 
     * StringParameterType supports definition of minimum and maximum lengths that can be used to validate the input received 
     * for the string parameter. It also supports definition of a list of valid input values.
     * See {@link StringParameter}
     * 
     * @return String input parameters.
     */
    Collection<StringParameter> getStringParameters();
	
    /**
     * Gets the URI parameters.
     * 
     * When the default value attribute is specified for a URI parameter, its value MUST be a valid Uniform Resource Identifier. 
     * See the BaseParameterType ({@link Parameter}) section for details of the inherited attributes and elements.
     * See {@link URIParameter}
     *  
     * @return Universal Resource Identifier input parameters. 
     */
    Collection<URIParameter> getURIParameters();
	
	/**
	 * Sets the boolean parameters.
	 * 
	 * BooleanParameterType extends BaseParameterType without adding any additional attributes or elements. When the defaultValue 
     * attribute is defined for a boolean parameter, its value MUST be either true or false.
     * See {@link BooleanParameter}
     * 
	 * @param booleanParameters Boolean input parameters.
	 */
	void setBooleanParameters(Collection<BooleanParameter> booleanParameters);
	
	/**
	 * Sets the integer parameters.
	 * 
	 * IntegerParameterType defines upper and lower bounds that can be used to validate the input received for that parameter.
     * See {@link IntegerParameter}
     * 
	 * @param integerParameters Integer input parameters.
	 */
	void setIntegerParameters(Collection<IntegerParameter> integerParameters);
	
	/**
	 * Sets the string parameters.
	 * 
	 * StringParameterType supports definition of minimum and maximum lengths that can be used to validate the input received 
     * for the string parameter. It also supports definition of a list of valid input values.
     * See {@link StringParameter}
     * 
	 * @param stringParameters String input parameters.
	 */
	void setStringParameters(Collection<StringParameter> stringParameters);
	
	/**
	 * Sets the URI parameters.
	 * 
	 * When the default value attribute is specified for a URI parameter, its value MUST be a valid Uniform Resource Identifier. 
     * See the BaseParameterType ({@link Parameter}) section for details of the inherited attributes and elements.
     * See {@link URIParameter}
     *  
	 * @param uriParameters Universal Resource Identifier input parameters. 
	 */
	void setURIParameters(Collection<URIParameter> uriParameters);
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	Object clone();
}
