/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * A VariableFactory is used to create objects representing or directly
 * related to variable.
 *
 */
public interface VariableFactory {
	/**
	 * Creates a new ParameterGroup.
	 * 
	 * @return A new ParameterGroup
	 */
    ParameterGroup createParameterGroup();
    
    /**
     * Creates a new IntegerParameter.
     * 
     * @param parameterID Value to use for the id attribute of the IntegerParameter
     * @return A new IntegerParameter
     * @throws InvalidInputException if parameterID is null
     */
    IntegerParameter createIntegerParameter(String parameterID) throws InvalidInputException;
    
    /**
     * Creates a new StringParameter.
     * 
     * @param parameterID Value to use for the id attribute of the StringParameter
     * @return A new StringParameter
     * @throws InvalidInputException if parameterID is null
     */
    StringParameter createStringParameter(String parameterID) throws InvalidInputException;

    /**
     * Creates a new URIParameter.
     * 
     * @param parameterID Value to use for the id attribute of the URIParameter
     * @return A new URIParameter
     * @throws InvalidInputException if parameterID is null
     */
    URIParameter createURIParameter(String parameterID) throws InvalidInputException;
    
    /**
     * Creates a new BooleanParameter.
     * 
     * @param parameterID Value to use for the id attribute of the BooleanParameter
     * @return A new BooleanParameter
     * @throws InvalidInputException if parameterID is null
     */
    BooleanParameter createBooleanParameter(String parameterID) throws InvalidInputException;
    
    /**
     * Creates a new ResourceProperty.
     * 
     * @param variableID Value to use for the id attribute of the ResourceProperty
     * @param resourceReference Value to use for the resourceRef attribute of the ResourceProperty
     * @return A new ResourceProperty
     * @throws InvalidInputException if variableID or resourceReference is null
     */
    ResourceProperty createResourceProperty(String variableID, String resourceReference) throws InvalidInputException;
    
    /**
     * Creates a new ConditionalDerivedVariable.
     * 
     * @param variableID Value to use for the id attribute of the ConditionalDerivedVariable
     * @param cdvExpressions
     * @return A new ConditionalDerivedVariable
     * @throws InvalidInputException if variableID is null, or if cdvExpressions is null or empty
     */
    ConditionalDerivedVariable createConditionalDerivedVariable(
            String variableID, Collection<ConditionalDerivedVariableExpression> cdvExpressions)
    		throws InvalidInputException;
    
    /**
     * Creates a new ConditionalDerivedVariableExpression.
     * 
     * @param expression Value to use for the Expression element
     * @param condition Value to use for the Condition element
     * @return A new ConditionalDerivedVariableExpression
     * @throws InvalidInputException if expression or condition is null
     */
    ConditionalDerivedVariableExpression createConditionalDerivedVariableExpression(
            String expression, Condition condition) throws InvalidInputException;
    
    /**
     * Creates a new DerivedVariable.
     * 
     * @param variableID Value to use for the id attribute of the DerivedVariable
     * @param expression Value to use for the Expression element of the DerivedVariable
     * @return A new DerivedVariable with element name "DerivedVariable"
     * @throws InvalidInputException if variableID or expression is null
     */
    DerivedVariable createDerivedVariable(String variableID, String expression) throws InvalidInputException;

    /**
     * Creates a new DerivedVariable.
     * 
     * @param variableID Value to use for the id attribute of the DerivedVariable
     * @param expression Value to use for the Expression element of the DerivedVariable
     * @return A new DerivedVariable with element name "Variable"
     * @throws InvalidInputException if variableID or expression is null
     */
    DerivedVariable createVariable(String variableID, String expression) throws InvalidInputException;

    /**
     * Creates a new ParameterBoundary.
     * 
     * @return a New ParameterBoundary
     */
    ParameterBoundary createParameterBoundary();
    
    /**
     * Creates a new OutputVariable.
     * 
     * @param id Value to use for the id attribute of the BooleanParameter
     * @return A new OutputVariable
     * @throws InvalidInputException if id is null
     */
    OutputVariable createOutputVariable(String id) throws InvalidInputException;
}