/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.provisional.cmdbf.query.client;

import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;

import org.apache.axis2.AxisFault;
import org.apache.axis2.Constants;
import org.apache.axis2.addressing.EndpointReference;
import org.apache.axis2.client.Options;
import org.apache.axis2.client.ServiceClient;
import org.apache.axis2.util.XMLUtils;

import org.eclipse.cosmos.dc.provisional.cmdbf.client.CMDBfServiceClient;
import org.eclipse.cosmos.dc.provisional.cmdbf.exception.CMDBfException;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.QueryOutputTransformer;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.input.artifacts.IQuery;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.query.transform.response.artifacts.IQueryResult;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.TransformerException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * Remote client for a CMDBf query service.  
 *
 */
public class QueryServiceClient extends CMDBfServiceClient {

	/**
	 * Creates a query service client, given the EPR of the service
	 * @param epr Endpoint reference URL of the registration service
	 */
	public QueryServiceClient(String epr) {
		super( epr );
	}
	
	/**
	 * Creates a query service client, given the EnpointReference object
	 * @param epr EndpointReference object representing the query service EPR
	 */
	public QueryServiceClient(EndpointReference epr) {
		super( epr );
	}
	
	/**
	 * graph query operation of the query service
	 * @param queryRequest query request in AXIOM object model
	 * @return query response in AXIOM object model
	 * @throws CMDBfException If error occurred during query
	 */
	public OMElement graphQuery(OMElement queryRequest) throws CMDBfException {
		try {
	        Options options = new Options();
	        options.setTo(serviceEPR);
	        options.setTransportInProtocol(Constants.TRANSPORT_HTTP);
	        
	        if ( soapVersion == 11 )
		        options.setSoapVersionURI(org.apache.axiom.soap.SOAP11Constants.SOAP_ENVELOPE_NAMESPACE_URI);
	        else
	        	options.setSoapVersionURI(org.apache.axiom.soap.SOAP12Constants.SOAP_ENVELOPE_NAMESPACE_URI);
	
	        ServiceClient sender = new ServiceClient();
	        
	        //
	        // Basic security?
	        //
	        if ( username != null && password != null ){
	        	//
	        	// construct security header and add to SOAP header...
	        	//
	        	addBasicSecurityHeader( sender, username, password);
	        }
	        
	        sender.setOptions(options);
	        OMElement result = sender.sendReceive(queryRequest);
	        return result;
		} catch (AxisFault e) {
			throw new CMDBfException(e);
		}
	}
	
	/**
	 * graph query operation of the query service, where input and output 
	 * are represented in document object model (DOM). 
	 * @param queryRequest query request in DOM Element object
	 * @return query response in DOM Element object
	 * @throws CMDBfException If error occurred during query
	 */
	public Element graphQuery(Element queryRequest) throws CMDBfException {
		try {
			OMElement request = XMLUtils.toOM(queryRequest);
			OMElement result = graphQuery(request);
			return XMLUtils.toDOM(result);
		} catch (CMDBfException e) {
			throw e;
		} catch (Exception e) {
			throw new CMDBfException(e);
		}
	}
	
	/**
	 * graph query operation of the query service, where input and output 
	 * are represented in COSMOS object model. 
	 * 
	 * @param queryRequest query request
	 * @return query response
	 * @throws CMDBfException If error occurred during query
	 */
	public IQueryResult graphQuery(IQuery queryRequest) throws CMDBfException {
		
		try {
			StringWriter queryWriter = new StringWriter();
			queryRequest.toXML(queryWriter, 0);
			StringReader queryReader = new StringReader(queryWriter.toString());
			
			//create the parser
			XMLStreamReader parser = XMLInputFactory.newInstance().createXMLStreamReader(queryReader);
			//create the builder
			StAXOMBuilder builder = new StAXOMBuilder(parser);
	
			//get the root element
			OMElement requestPayload = builder.getDocumentElement();
			
			OMElement result = graphQuery(requestPayload);

	        IQueryResult resultObj = QueryOutputTransformer.transform(new StringReader(result.toString()));
	        
	        return resultObj;
		} catch (XMLStreamException e) {
			throw new CMDBfException(e);
		} catch (TransformerException te) {
			throw new CMDBfException(te);
		}
	}
	
	protected Element getWSDL() throws CMDBfException {
		try {
			String wsdlurl = serviceEPR.getAddress() + "?wsdl";
			URL url = new URL(wsdlurl);
			URLConnection connection = url.openConnection();
			HttpURLConnection httpConn = (HttpURLConnection) connection;
			// Set the appropriate HTTP parameters.
			httpConn.setRequestProperty("Content-Type", "text/xml; charset=utf-8");
			httpConn.setRequestMethod("GET");
			InputStream input = httpConn.getInputStream();
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			factory.setNamespaceAware(true);
			Document doc = factory.newDocumentBuilder().parse(input);
            Element elem = doc.getDocumentElement();
            return elem;
		} catch (Exception e) {
			throw new CMDBfException(e);
		}
	}

	/**
	 * Gets the service metadata of the query service. 
	 * @return metadata definition represented in DOM Element format
	 * @throws CMDBfException
	 */
	public Element getQueryServiceMetadata() throws CMDBfException {
		Element metadata = null;
		Element wsdl = getWSDL();
        NodeList list = wsdl.getElementsByTagNameNS("http://cmdbf.org/schema/1-0-0/serviceMetadata", "queryServiceMetadata");
        if (list.getLength() > 0) {
        	// assume there is only one policy for query service metadata
        	metadata = (Element) list.item(0);
        }
		return metadata;
	}
	
}
