/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.tests.common.spi;

import java.math.BigInteger;
import java.util.Arrays;
import java.util.Properties;

import junit.framework.TestCase;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.SPISession;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ConstrainedResource;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Multiplicity;
import org.w3c.dom.Element;

public class Multiplicity_Test extends TestCase {
    Multiplicity multiplicity;
    Multiplicity clonedMultiplicity;
    Element element;
    SPISessionImpl session;

    public Multiplicity_Test(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
        
        multiplicity = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createMultiplicity(Boolean.TRUE);
        ConstrainedResource conRes1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createConstrainedResource("resource1");
        ConstrainedResource conRes2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createConstrainedResource("resource2");
        
        multiplicity.setConstrainedFeatures(Arrays.asList("feature1", "feature2"));
        multiplicity.setConstrainedResources(Arrays.asList(conRes1, conRes2));
        multiplicity.setMaxSelections(BigInteger.valueOf(5));
        multiplicity.setMinSelections(BigInteger.valueOf(4));
        
        clonedMultiplicity = (Multiplicity)multiplicity.clone();
        element = ((SPIDataObject)multiplicity).getElement();
        session = (SPISessionImpl)SPISession.DEFAULT_INSTANCE;
    }

    public void testGetTranslatableStrings() {
        Properties props = ((SPIDataObject) multiplicity).getTranslatableStrings();
        
        assertEquals(0, props.keySet().size());
    }

    public void testGetConstrainedFeatures() {
        assertEquals(2, multiplicity.getConstrainedFeatures().size());
        assertEquals(2, clonedMultiplicity.getConstrainedFeatures().size());
        assertEquals(2, session.getChildren(element, "ConstrainedFeature").size());
        
        assertEquals("feature1", multiplicity.getConstrainedFeatures().toArray(new String[0])[0]);
        assertEquals("feature2", multiplicity.getConstrainedFeatures().toArray(new String[0])[1]);
        
        assertEquals("feature1", clonedMultiplicity.getConstrainedFeatures().toArray(new String[0])[0]);
        assertEquals("feature2", clonedMultiplicity.getConstrainedFeatures().toArray(new String[0])[1]);

        assertEquals("feature1", ( session.getChildren(element, "ConstrainedFeature").toArray(new Element[0])[0]).getAttribute("featureRef"));
        assertEquals("feature2", ( session.getChildren(element, "ConstrainedFeature").toArray(new Element[0])[1]).getAttribute("featureRef"));
    }

    public void testGetConstrainedResources() {
        assertEquals(2, multiplicity.getConstrainedResources().size());
        assertEquals(2, clonedMultiplicity.getConstrainedResources().size());
        assertEquals(2, session.getChildren(element, "ConstrainedResource").size());
        
        assertEquals("resource1", multiplicity.getConstrainedResources().toArray(new ConstrainedResource[0])[0].getResourceRef());
        assertEquals("resource2", multiplicity.getConstrainedResources().toArray(new ConstrainedResource[0])[1].getResourceRef());
        
        assertEquals("resource1", clonedMultiplicity.getConstrainedResources().toArray(new ConstrainedResource[0])[0].getResourceRef());
        assertEquals("resource2", clonedMultiplicity.getConstrainedResources().toArray(new ConstrainedResource[0])[1].getResourceRef());

        assertEquals("resource1", ( session.getChildren(element, "ConstrainedResource").toArray(new Element[0])[0]).getAttribute("resourceRef"));
        assertEquals("resource2", ( session.getChildren(element, "ConstrainedResource").toArray(new Element[0])[1]).getAttribute("resourceRef"));
    }

    public void testGetMaxSelections() {
        assertEquals(5, multiplicity.getMaxSelections().intValue());
        assertEquals(5, clonedMultiplicity.getMaxSelections().intValue());
        assertEquals(5, Integer.parseInt(element.getAttribute("maxSelections")));
    }

    public void testGetMinSelections() {
        assertEquals(4, multiplicity.getMinSelections().intValue());
        assertEquals(4, clonedMultiplicity.getMinSelections().intValue());
        assertEquals(4, Integer.parseInt(element.getAttribute("minSelections")));
    }

    public void testGetMultiplesAllowed() {
        assertTrue(multiplicity.getMultiplesAllowed().booleanValue());
        assertTrue(clonedMultiplicity.getMultiplesAllowed().booleanValue());
        assertTrue(Boolean.parseBoolean(element.getAttribute("multiplesAllowed")));
    }

    public void testSetConstrainedFeatures() {
        multiplicity.setConstrainedFeatures(Arrays.asList("feature3"));
        
        assertEquals(1, multiplicity.getConstrainedFeatures().size());
        assertEquals(1, session.getChildren(element, "ConstrainedFeature").size());
        
        assertEquals("feature3", multiplicity.getConstrainedFeatures().toArray(new String[0])[0]);
        assertEquals("feature3", ( session.getChildren(element, "ConstrainedFeature").toArray(new Element[0])[0]).getAttribute("featureRef"));
    }

    public void testSetConstrainedResources() {
    	ConstrainedResource conRes3 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createConstrainedResource("resource3");
        multiplicity.setConstrainedResources(Arrays.asList(conRes3));

        assertEquals(1, multiplicity.getConstrainedResources().size());
        assertEquals(1, session.getChildren(element, "ConstrainedResource").size());
        
        assertEquals("resource3", multiplicity.getConstrainedResources().toArray(new ConstrainedResource[0])[0].getResourceRef());
        assertEquals("resource3", ( session.getChildren(element, "ConstrainedResource").toArray(new Element[0])[0]).getAttribute("resourceRef"));
    }

    public void testSetMaxSelections() {
        multiplicity.setMaxSelections(BigInteger.valueOf(6));
        
        assertEquals(6, multiplicity.getMaxSelections().intValue());
        assertEquals(6, Integer.parseInt(element.getAttribute("maxSelections")));
    }

    public void testSetMinSelections() {
        multiplicity.setMinSelections(BigInteger.valueOf(2));
        
        assertEquals(2, multiplicity.getMinSelections().intValue());
        assertEquals(2, Integer.parseInt(element.getAttribute("minSelections")));
    }

    public void testSetMultiplesAllowed() {
        multiplicity.setMultiplesAllowed(Boolean.FALSE);
        
        assertFalse(multiplicity.getMultiplesAllowed().booleanValue());
        assertFalse(Boolean.parseBoolean(element.getAttribute("multiplesAllowed")));
        
        try {
            multiplicity.setMultiplesAllowed(null);
            fail("MulitplesAllowed set to null");
        }
        catch (Exception e) {
        }
    }
}
