/**********************************************************************
 * Copyright (c) 2008, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.artifacts;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.w3c.dom.Node;

/**
 * This structure is used to store DOM representation of instance
 * documents.  Each DOM node is keyed based on the alias of the
 * document.  If in case a document doesn't have an alias, it
 * can be retrieved using getOrphanedDocuments().
 * 
 * @author Ali Mehregani
 * @author David Whiteman
 * @author John Arwe
 */
public class DOMStructure
{
	/**
	 * The document nodes are keyed based on the alias of the 
	 * document.  If a document happens to have multiple aliases, 
	 * there will be multiple entries in this map.
	 */
	private Map<String, Node> documents;
	
	/**
	 * The document nodes are keyed based on the alias of the 
	 * document.  If a document happens to have multiple aliases, 
	 * there will be multiple entries in this map.
	 */
	private Map<Node, String[] > aliasesByDocumentNode;
	
	/**
	 * Stores document nodes that have no aliases
	 */
	private List<Node> orphanedDocuments;
	
	/**
	 * Map of line numbers to elements 
	 */
	private Map<Node, ElementLocation> lineNumberMap;

	public DOMStructure()
	{
		documents = new HashMap<String, Node>();
		aliasesByDocumentNode = new HashMap<Node, String[]>();
		orphanedDocuments = new ArrayList<Node>();
		lineNumberMap = new HashMap<Node, ElementLocation>();
	}

	
	/**
	 * Retrieve and return the DOM node associated with
	 * the document alias passed in
	 * 
	 * @param alias A document alias
	 * @return The associated DOM node
	 */
	public Node get(String alias)
	{
		return documents.get(alias);
	}
	
	/**
	 * Retrieve and return the document aliases associated with
	 * the DOM document node passed in
	 * 
	 * @param alias A document alias
	 * @return The associated DOM node
	 */
	public String[] getAliases(Node document)
	{
		return aliasesByDocumentNode.get(document);
	}
	
	public ElementLocation getLocation(Node node) {
		return lineNumberMap.get(node);
	}
	
	public List<Node> getOrphanedDocuments() {
		return orphanedDocuments;
	}
	
	public Map<String, Node> getDocuments() {
		return documents;
	}
	
	/**
	 * Adds an entry to this structure.
	 * 
	 * @param documentNode A document node
	 * @param aliases The aliases of the document
	 */
	public void add(Node documentNode, String[] aliases)
	{
		if (aliases == null || aliases.length <= 0)
		{
			orphanedDocuments.add(documentNode);
			return;
		}
		
		for (int i = 0; i < aliases.length; i++)
		{
			documents.put(aliases[i], documentNode);
		}
		aliasesByDocumentNode.put(documentNode.getOwnerDocument(), aliases);
	}
	
	public void add(Node documentNode, ElementModel location)
	{
		add(documentNode, location.getAliases());
		lineNumberMap.put(documentNode, location);
	}
	
	/**
	 * Retrieve the node for the orphaned document (i.e. a 
	 * document containing no alias) with the index passed in.
	 * 
	 * @param index The index to lookup with
	 * @return The node of the i-th orphaned document
	 */
	public Node getOrphan(int index)
	{
		return index >= 0 && index < orphanedDocuments.size() ? orphanedDocuments.get(index) : null;
	}
	
	
	/**
	 * Retrieve the index for the orphaned document (i.e. a 
	 * document containing no alias) given a document node.
	 * 
	 * @param document The document node to search for in the orphaned documents list
	 * @return index The orphan index of the document
	 */
	public int getOrphan(Node document)
	{
		int index = -1;
		for (int i = 0; i < orphanedDocuments.size(); i++)
		{
			if (orphanedDocuments.get(i) == document)
			{
				index = i;
			}
		}
		return index ;
	}
	
	
	/**
	 * Returns the length of the stored entries
	 *  
	 * @return The size of this data structure
	 */
	public int size()
	{
		return documents.size() + orphanedDocuments.size();
	}
}
