/*******************************************************************************
 * Copyright (c) 2009 SAS Institute, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SAS Institute - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.sdd.cim.profile;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.util.StringTokenizer;

import org.eclipse.cosmos.me.sdd.cr.Constraint;
import org.eclipse.cosmos.me.sdd.cr.IResolver;
import org.eclipse.cosmos.me.sdd.schema.ext.ResolutionObject;

/**
 * Class to resolve properties and constraints placed on FileSystem types.
 * 
 * @author jehamm
 *
 */
public class FileSystemResolver implements IResolver {

	/**
	 * Resolve this class based on Constraint defined. The Constraint may define a property or
	 * a version.
	 * 
	 * @param c
	 */
	public ResolutionObject resolve(Constraint c) {
		// Create a container for resolution
		ResolutionObject resolution = new ResolutionObject();
		if(c.getType().equalsIgnoreCase("property")) {
			return resolveProperty(c, resolution);
		}
		else if(c.getType().equalsIgnoreCase("version")) {
			return resolveVersion(c, resolution);
		}
		else if(c.getType().equalsIgnoreCase("capacity")) {
			return resolveCapacity(c, resolution);
		}
		else if(c.getType().equalsIgnoreCase("consumption")) {
			return resolveConsumption(c, resolution);
		}
		return resolution;
	}
	
	/**
	 * Resolve a consumption constraint
	 * 
	 * @param c
	 * @param resolution
	 * @return
	 */
	private ResolutionObject resolveConsumption(Constraint c, ResolutionObject resolution) {
		//AvailableSpace
		
		for(String name : c.getNames()) {
			if(name.endsWith("AvailableSpace")) {
				return resolveFileSystemAvailableSpace(name, c, resolution);
			}
		}
		
		return resolution;
	}

	/**
	 * Resolve a capacity constraint.
	 * 
	 * @param c
	 * @param resolution
	 * @return
	 */
	private ResolutionObject resolveCapacity(Constraint c, ResolutionObject resolution) {
		System.out.println("FileSystemResolver.resolveCapacity() - not implemented.");
		return resolution;
	}


	/**
	 * Resolve version data that can be associated with CIM_OperatingSystem
	 * @param c
	 * 		Constraint instance that defines the SDD property constraints.
	 * @param resolution
	 * 		Resolution instance that will contain the results of the resolution.
	 * @return
	 * 		the completed resolution if we found a 
	 */
	private ResolutionObject resolveVersion(Constraint c, ResolutionObject resolution) {
		System.out.println("FileSystemResolver.resolveVersion() - not implemented.");
		return resolution;
	}

	/**
	 * Resolve property that can be associated with CIM_OperatingSystem
	 * @param c
	 * 		Constraint instance that defines the SDD property constraints.
	 * @param resolution
	 * 		Resolution instance that will contain the results of the resolution.
	 * @return
	 * 		the completed resolution if we found a 
	 */
	private ResolutionObject resolveProperty(Constraint c, ResolutionObject resolution) {
		System.out.println("FileSystemResolver.resolveProperty() - not implemented.");
		return resolution;
	}

	/**
	 * Detailed resolver for checking file system space.
	 * @param name
	 * @param c
	 * @param resolution
	 */
	private ResolutionObject resolveFileSystemAvailableSpace(String name, Constraint c,
			ResolutionObject resolution) {
		
		String directory = c.getResource();
		File f = new File(directory);

		if (f.exists() && f.isDirectory()) {
			
			// See how much space is actually being asked for.
			//TODO -- need units from the SDD here to calculate the size
			
			String [] values = c.getValues("CIM_FileSystem.AvailableSpace");
			long requestedSize = Long.parseLong(values[0]);
			requestedSize = requestedSize * 512;
			
			// See what we have available.
			long freeSpace = getFreeSpace(directory);
			
			// Set the appropriate outcome.
			if(freeSpace > requestedSize) {
				resolution.accept();
				c.accept();
			}
			else {
				resolution.reject();
				c.reject();
			}
		}
			
		return resolution;
	}

	/**
	 * Try and locate the amount of space on the requested machine.
	 */ 
	private long getFreeSpace(String directoryName) { 
		try 
		{ 
			// Get the machine we are on.
			String os = System.getProperty("os.name"); 
			String command; 
			
			// Check the operating syteem.
			if (os.contains("Windows")) { 
				command = "cmd.exe /c dir \"" + directoryName + "\""; 
			} 
			else { 
				command = "df " + directoryName; 
			} 
			
			// Run the command.
			Runtime runtime = Runtime.getRuntime(); 
			Process process = runtime.exec(command); 
			
			
			if (process == null) { 
				return 0; 
			} 
			
			//process.waitFor();
			
			// Read the output of the command.
			BufferedReader inStream = new BufferedReader(new InputStreamReader(process.getInputStream())); 
			String line = null; 
			String output = null;
			while ((line = inStream.readLine()) != null) { 
				output = line; 
			} 
			
			// Determine which one to run...
			if (os.contains("Windows")) { 
				return windowsFreeSpace(output);
			} 
			else { 
				return unixFreeSpace(output);
			}  
		} 
		catch (Exception e) { 
			return 0; 
		} 
	}
	
	/**
	 * 
	 * @param output
	 * @return
	 */
	private long unixFreeSpace(String output) {
		
		// Make sure we got something.
		if (output != null) {		
			// Alter the line so that we are looking at the right content.
			String availableSpace = output.trim(); 
			StringTokenizer outputTokens = new StringTokenizer(output, ":");
			
			// skip the first token.
			outputTokens.nextToken();
			availableSpace = outputTokens.nextToken();
			String[] tokens = availableSpace.split(" "); 
			
			//The first valid value in items is the free space.
			int idx = 1; 
			while(idx < tokens.length) { 
				try { 
					long total = Long.parseLong(tokens[idx++].trim()); 
					return total; 
				} 
				catch (NumberFormatException e) { 
					return 0;
				} 
			} 
		}
		return 0;
	}
	private long windowsFreeSpace(String output) {
		
		// Make sure we got something.
		if (output != null) {		
			// Alter the line so that we are looking at the right content.
			String availableSpace = output.trim(); 
			availableSpace = availableSpace.replaceAll("\\.", ""); 
			availableSpace = availableSpace.replaceAll(",", ""); 
			String[] tokens = availableSpace.split(" "); 
			
			//The first valid value in items is the free space.
			int idx = 1; 
			while(idx < tokens.length) { 
				try { 
					long total = Long.parseLong(tokens[idx++]); 
					return total; 
				} 
				catch (NumberFormatException e) { 
					//return 0;
				} 
			} 
		}
		return 0;
	}
}
