/*******************************************************************************
 * Copyright (c) 2009 SAS Institute, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SAS Institute - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.sdd.cim.profile;

import org.eclipse.cosmos.me.sdd.cr.Constraint;
import org.eclipse.cosmos.me.sdd.cr.IResolver;
import org.eclipse.cosmos.me.sdd.schema.ext.ResolutionObject;

public class OperatingSystemResolver implements IResolver {

	/**
	 * Resolve this class based on Constraint defined. The Constraint may define a property or
	 * a version.
	 * 
	 * @param c
	 */
	public ResolutionObject resolve(Constraint c) {
		// Create a container for resolution
		ResolutionObject resolution = new ResolutionObject();
		if(c.getType().equalsIgnoreCase("property")) {
			return resolveProperty(c, resolution);
		}
		else if(c.getType().equalsIgnoreCase("version")) {
			return resolveVersion(c, resolution);
		}
		else if(c.getType().equalsIgnoreCase("capacity")) {
			return resolveCapacity(c, resolution);
		}
		else if(c.getType().equalsIgnoreCase("consumption")) {
			return resolveConsumption(c, resolution);
		}
		return resolution;
	}
	
	private ResolutionObject resolveConsumption(Constraint c, ResolutionObject resolution) {
		System.out.println("OperatingSystemResolver.resolveConsumption() - not implemented.");
		return resolution;
	}

	private ResolutionObject resolveCapacity(Constraint c, ResolutionObject resolution) {
		System.out.println("OperatingSystemResolver.resolveCapacity() - not implemented.");
		return resolution;
	}

	/**
	 * Resolve version data that can be associated with CIM_OperatingSystem
	 * @param c
	 * 		Constraint instance that defines the SDD property constraints.
	 * @param resolution
	 * 		Resolution instance that will contain the results of the resolution.
	 * @return
	 * 		the completed resolution if we found a 
	 */
	private ResolutionObject resolveVersion(Constraint c, ResolutionObject resolution) {
		for(String name : c.getNames()) {
			resolveOSVersion(name, c, resolution);
		}
		
		return resolution;
	}

	/**
	 * Resolve property that can be associated with CIM_OperatingSystem
	 * @param c
	 * 		Constraint instance that defines the SDD property constraints.
	 * @param resolution
	 * 		Resolution instance that will contain the results of the resolution.
	 * @return
	 * 		the completed resolution if we found a 
	 */
	private ResolutionObject resolveProperty(Constraint c, ResolutionObject resolution) {
		//OSType
		//Version
		//NumberOfProcesses
		//Hosts
		for(String name : c.getNames()) {
			if(name.endsWith("OSType")) {
				resolveOSType(name, c, resolution);
			}
			else if(name.endsWith("Version")) {
				resolveOSVersion(name, c, resolution);
			}
		}
		
		return resolution;
	}

	/**
	 * Resolve an operating system type constraint
	 */
	private ResolutionObject resolveOSType(String name, Constraint c, ResolutionObject resolution) {
		
		// Get the working operation system name.
		String osName = System.getProperty("os.name");
		
		// If there are a list of values, then make sure that the current OS is one of them.
		String[] values = c.getValues(name);
		for(int i = 0; i < values.length; i++) {
			String val = values[i];
			if(val.equalsIgnoreCase(osName)) {
				//resolution.addResolvedResourceProperty(resourceId, properties.get("name"));
				resolution.accept();
				c.accept();
			}
			else {
				resolution.reject();
				c.reject();
			}
		}
		
		return resolution;
	}
	
	/**
	 * Resolve an operating system version constraint.
	 */
	private ResolutionObject resolveOSVersion(String name, Constraint c, ResolutionObject resolution) {
		
		// Get the working operating system version information.
		String osVersion = System.getProperty("os.version");	
		float version = convertVersion(osVersion);
		
		boolean passes = false;
		// Is this version check have a min/max or both.
		if(name.equals("minimum")) {
			// Convert the value defined in the SDD.
			String[] values = c.getValues(name);
			float sdd_value = convertVersion(values[0]);
			if(sdd_value <= version) {
				passes = true;
			}
		}
		else if(name.equalsIgnoreCase("maximum")) {
			
			// Convert the value defined in the SDD.
			String[] values = c.getValues(name);
			float sdd_value = convertVersion(values[0]);
			if(sdd_value >= version) {
				passes = true;
			}
		}
			
		// Check to see if we made it through the version checking.
		if(passes) {
			//resolution.addResolvedResourceProperty(resourceId, properties.get("name"));
			resolution.accept();
			c.accept();
		}
		else {
			resolution.reject();
			c.reject();
		}
		
		return resolution;
	}
	
	/**
	 * Function to help convert pieces of version.
	 * @param version
	 * @return
	 */
	private float convertVersion( String version )
	{
		String pieces[] = version.split( "\\." );
		if ( pieces.length == 1 )
			return Float.parseFloat( pieces[0] );
		else 
			return Float.parseFloat( pieces[0] + "." + pieces[1] );
	}
}
