/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.operations;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.rm.internal.repository.RepositoryMessages;
import org.eclipse.cosmos.rm.internal.validation.artifacts.RuleBinding;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil.RemoteRetrievalException;
import org.eclipse.osgi.util.NLS;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * SAX Handler for parsing an SML-IF file for definition and instance model units
 * 
 * @author David Whiteman
 * @author Ali Mehregani
 */
public class SMLIFFileHandler extends DefaultHandler 
{
	/**
	 * Boolean flags used to indicate the section of the document that 
	 * is currently being parsed.
	 */
	private boolean inData, inDefinitions, inAlias, 
					inBaseURI, inDescription, inName, 
					inDisplayName, inDocumentAlias, inRuleAlias, 
					inVersion;
	
	private boolean buildingModelUnit;
	private ModelUnit currentModel;
	
	private List<ModelUnit> modelUnits;
	private List<RuleBinding> ruleBindingMappings;
	
	private StringBuffer currentElementBuffer, currentRuleAliasBuffer, 
						 currentDocumentAliasBuffer;
	
	private String 	baseURI, name, description,
					displayName, version;

	
	/**
	 * A boolean to indicate that the first element after {@link ISMLConstants#DATA_ELEMENT}
	 * has been reached.
	 */
	private boolean firstElement;
	
	/**
	 * The definition files indexed by their target namespace
	 * KEY = A {@link String} representing the target namespace
	 * VALUE = A {@link String} represeting the file name of the model unit
	 */
	private Map<String, String> definitionFiles;
	
	/**
	 * Stores the current target namespace of the definition file
	 * detected.
	 */
	private String currentTargetNamespace;

	
	public SMLIFFileHandler()
	{
		modelUnits = new ArrayList<ModelUnit>();
		ruleBindingMappings = new ArrayList<RuleBinding>();
		currentElementBuffer = new StringBuffer();
		currentRuleAliasBuffer = new StringBuffer();
		currentDocumentAliasBuffer = new StringBuffer();
		definitionFiles = new Hashtable<String, String>();		
	}
	
	
	public void characters(char[] ch, int start, int length) throws SAXException {
		String elementData = new String(ch, start, length);
		if (inData) {
			writeToModelUnitFile(elementData);
		}
		String trimmedElementData = elementData.trim();
		if (inAlias || inBaseURI || inName  || inDisplayName || inVersion) {
			currentElementBuffer.append(trimmedElementData);
		} else if (inDocumentAlias) {
			currentDocumentAliasBuffer.append(trimmedElementData);
		} else if (inRuleAlias) {
			currentRuleAliasBuffer.append(trimmedElementData);
		}
		if (inDescription) {
			currentElementBuffer.append(elementData);
		}
		super.characters(ch, start, length);
	}
	
	public void endElement(String uri, String localName, String qName) throws SAXException {
		if (ISMLConstants.DEFINITIONS_ELEMENT.equals(localName)) 
		{
			inDefinitions = false;
		}
		else if (ISMLConstants.DATA_ELEMENT.equals(localName)) 
		{
			inData = false;
			currentModel.stop();
			modelUnits.add(currentModel);
			currentModel = null;
		}
		else if (ISMLConstants.LOCATOR_ELEMENT.equals(localName)) 
		{
			if (currentModel != null) {
				modelUnits.add(currentModel);
				currentModel = null;
			}
		}
		else if (ISMLConstants.BASE_URI_ELEMENT.equals(localName)) 
		{
			inBaseURI = false;
			baseURI = currentElementBuffer.toString().trim();
			currentElementBuffer = new StringBuffer();
		}
		else if (ISMLConstants.NAME_ELEMENT.equals(localName)) 
		{
			inName = false;
			name = currentElementBuffer.toString().trim();
			currentElementBuffer = new StringBuffer();
		}
		else if (ISMLConstants.DISPLAY_NAME_ELEMENT.equals(localName)) 
		{
			inDisplayName = false;
			displayName = currentElementBuffer.toString().trim();
			currentElementBuffer = new StringBuffer();
		}
		else if (ISMLConstants.VERSION_ELEMENT.equals(localName)) 
		{
			inVersion = false;
			version = currentElementBuffer.toString().trim();
			currentElementBuffer = new StringBuffer();
		}
		else if (ISMLConstants.DESCRIPTION_ELEMENT.equals(localName)) 
		{
			inDescription = false;
			description = currentElementBuffer.toString();
			currentElementBuffer = new StringBuffer();
		}
		else if (ISMLConstants.ALIAS_ELEMENT.equals(localName)) 
		{
			inAlias = false;
			currentModel.addAlias(currentElementBuffer.toString().trim());
			currentElementBuffer = new StringBuffer();
		}
		else if (ISMLConstants.RULEBINDING_ELEMENT.equals(localName)) 
		{
			if (currentRuleAliasBuffer.length() > 0) {
				String docAlias = null;
				if (currentDocumentAliasBuffer.length() > 0) {
					docAlias = currentDocumentAliasBuffer.toString().trim();;
				}
				String ruleAlias = currentRuleAliasBuffer.toString().trim();
				ruleBindingMappings.add(new RuleBinding(docAlias, ruleAlias));
			}
			currentDocumentAliasBuffer = new StringBuffer();
			currentRuleAliasBuffer = new StringBuffer();
		} 
		else if (ISMLConstants.RULEALIAS_ELEMENT.equals(localName))
		{
			inRuleAlias = false;
		} 
		else if (ISMLConstants.DOCUMENTALIAS_ELEMENT.equals(localName)) 
		{
			inDocumentAlias = false;
		}
		
		if (inData) 
		{
	    	writeToModelUnitFile("</"+qName+">");
		}
		super.endElement(uri, localName, qName);
	}
	
	public String getModelLocation() {
		if (baseURI == null) {
			return "";
		}
		String result = baseURI;
		if (result.startsWith("http://")) {
			result = result.substring("http://".length());
			int nextSlash = result.indexOf('/');
			if (nextSlash == -1) {
				result = "";
			} else {
				result = result.substring(nextSlash) + File.separatorChar;
			}
		}
		return result;
	}
	
	public List<ModelUnit> getModelUnits() {
		return modelUnits;
	}
	
	public boolean inModelUnit() {
		return buildingModelUnit;
	}
	
	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException 
	{
		if (ISMLConstants.DOCUMENTALIAS_ELEMENT.equals(localName)) 
		{
			inDocumentAlias = true;
		}
		else if (ISMLConstants.RULEALIAS_ELEMENT.equals(localName)) 
		{
			inRuleAlias = true;
		}
		else if (ISMLConstants.DEFINITIONS_ELEMENT.equals(localName)) 
		{
			inDefinitions = true;
		}
		else if (ISMLConstants.BASE_URI_ELEMENT.equals(localName)) 
		{
			inBaseURI = true;
		}
		else if (ISMLConstants.NAME_ELEMENT.equals(localName)) 
		{
			inName = true;
		}
		else if (ISMLConstants.DISPLAY_NAME_ELEMENT.equals(localName)) 
		{
			inDisplayName = true;
		}
		else if (ISMLConstants.VERSION_ELEMENT.equals(localName)) 
		{
			inVersion = true;
		}
		else if (ISMLConstants.DESCRIPTION_ELEMENT.equals(localName)) 
		{
			inDescription = true;
		}
		else if (ISMLConstants.DOCUMENT_ELEMENT.equals(localName)) 
		{
			currentModel = new ModelUnit(inDefinitions ? ISMLConstants.TYPE_DEFINITION : ISMLConstants.TYPE_INSTANCE);
			currentModel.start();
		}
		else if (ISMLConstants.LOCATOR_ELEMENT.equals(localName)) 
		{
			// Need to retrieve remote document content
			String remoteDocUrlString = attributes.getValue(ISMLConstants.XLINK_URI, ISMLConstants.HREF_ATTRIBUTE);
			URL remoteDocUrl = null;
			try {
				remoteDocUrl = new URL(remoteDocUrlString);
			} catch (MalformedURLException e1) {
				// Bogus url
				currentModel = null;
				throw new SAXException(NLS.bind(RepositoryMessages.importErrorInvalidLocatorURL, remoteDocUrlString));
			}
			String localFileName = remoteDocUrl.getFile();
			currentModel.setFileName(localFileName);
			try {
				// TODO need to actually create a local file from the remote document
				// https://bugs.eclipse.org/bugs/show_bug.cgi?id=170822
				retrieveRemoteDocument(remoteDocUrlString);
			} catch (RemoteRetrievalException e) {
				// Can't find document
				currentModel = null;
				
				// where should this be logged?
				throw new SAXException(NLS.bind(RepositoryMessages.importErrorMissingFileAtLocatorURL, remoteDocUrlString));
			}
		}
		else if (ISMLConstants.ALIAS_ELEMENT.equals(localName)) 
		{
			// This helps us know we need to get the fileName data out of the aliases
			inAlias = true;
		}
		if (inData) 
		{
			buildingModelUnit = true;
			boolean isfirstInstanceElement = firstElement && !inDefinitions;
			Map<String, String> prefixes = isfirstInstanceElement ? new Hashtable<String, String>() : null;
			String schemaURIPrefix = null;
	        writeToModelUnitFile("<"+qName);
	        if (attributes != null) 
	        {
	            for (int i = 0; i < attributes.getLength(); i++) 
	            {
	                String aName = attributes.getQName(i);
	                writeToModelUnitFile(" ");
	                writeToModelUnitFile(aName+"=\""+attributes.getValue(i)+"\"");
	                
	                if (isfirstInstanceElement && ISMLConstants.SCHEMA_URI.equals(attributes.getValue(i)))
	                	schemaURIPrefix = attributes.getQName(i).replaceAll(attributes.getLocalName(i), "");                		                	
	            }
	        }
	        	        	        
	        /* Index the definition by its target namespace if we're in the definition section OR
	         * set the schemaLocation attribute of the first element if we're in the instances section. */ 
	        if (firstElement)
	        {
	        	currentModel.setFilePrefix(localName);
	        	/* We're in the definition section */
	        	if (inDefinitions)
	        	{
	        		currentTargetNamespace = attributes.getValue(ISMLConstants.TARGET_NAMESPACE_ATTRIBUTE);
	        		if (currentTargetNamespace != null)
	        		{
	        			String[] namespaceTokens = currentTargetNamespace.split("[/:]");
	        			if (namespaceTokens.length > 0) {
	        				// Use a name based on the target namespace if possible, using the rightmost
	        				// alphanumeric token in the name (e.g. "http://foo.com/core" 
	        				// becomes "core" and "urn:student" becomes "student")
	        				currentModel.setFilePrefix(namespaceTokens[namespaceTokens.length-1]);
	        			}
	        			currentModel.setFileName(currentModel.getFileName());
	        			definitionFiles.put(currentTargetNamespace, currentModel.getFileName());
	        		}
	        	}
	        	/* We're in the instances section */
	        	else if (uri != null)
	        	{
	        		String file = (String)definitionFiles.get(uri);
	        		if (file != null)
	        		{
	        			String prefix = schemaURIPrefix == null ? findUnboundPrefix(prefixes) : schemaURIPrefix;
	        			if (schemaURIPrefix == null)
	        			{
	        				/* We need to declare the schema namespace */
	        				writeToModelUnitFile(" " + ISMLConstants.XML_NS_ATTRIBUTE + ":" + prefix + "=\"" + ISMLConstants.SCHEMA_INSTANCE_URI + "\"");
	        			}
	        			String currentFileName = currentModel.getFileName();
	        			int currentInx = 0;
	        			String relativePath = "";
	        			while ((currentInx = currentFileName.indexOf('/', currentInx + 1)) > 0)
	        			{
	        				relativePath += "../";
	        			}
	        			
	        			writeToModelUnitFile(" " + prefix + ":" + ISMLConstants.SCHEMA_LOCATION_ATTRIBUTE + "=\"" + uri + " " + relativePath + file + "\"");
	        		}
	        			
	        	}
	        	firstElement = false;
	        }
	        
	        writeToModelUnitFile(">");
		}
		if (ISMLConstants.DATA_ELEMENT.equals(localName)) 
		{
			inData = true;
			firstElement = true;
		}
		super.startElement(uri, localName, qName, attributes);
	}
	
	private String findUnboundPrefix(Map<String, String> prefixes)
	{
		String prefix = "xsi";
		while (prefixes.get(prefix) != null)
			prefix += "i";
		
		return prefix;
	}

	/**
	 * Retrieve the document referenced by remoteDocUrlString and
	 * set the contents of the current model unit accordingly.
	 * 
	 * @param remoteDocUrlString
	 * @throws RemoteRetrievalException
	 */
	private void retrieveRemoteDocument(String remoteDocUrlString) throws RemoteRetrievalException {
		URL locator;
		Exception exception = null;
		try {
			locator = new URL(remoteDocUrlString);
			InputStream stream = locator.openStream();
			BufferedReader reader = new BufferedReader(new InputStreamReader(stream));
			String line;
			while ((line = reader.readLine()) != null) {
				currentModel.appendToContents(line+"\n");
			}
			reader.close();
			return;
		} catch (MalformedURLException e) {
			exception = e;
		} catch (IOException e) {
			exception = e;
		}
		throw new RemoteRetrievalException(exception);
	}
	
	private void writeToModelUnitFile(String string) {
		currentModel.appendToContents(string);
	}
	
	public String getBaseURI() {
		return baseURI;
	}

	public String getDescription() {
		return description;
	}

	public String getName() {
		return name;
	}

	public String getDisplayName() {
		return displayName;
	}

	public String getVersion() {
		return version;
	}

	public List<RuleBinding> getRuleBindingMappings() {
		return ruleBindingMappings;
	}

}
