/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Describable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * InternalDependencyType provides the type definition for Dependency elements defined in all types of content elements. Dependency 
 * elements allow the expression of dependence on the application of a particular operation to a content element defined in the 
 * deployment descriptor before application of a particular operation on the defining content element. The dependency is associated 
 * with an operation on the defining content element by the operation attribute in the Requirement defining the Dependency element. 
 * The dependency is associated with an operation on the depended on content element by the contentRefOperation attribute in the 
 * Dependency. There are three types of dependencies: pre-requisites, co-requisites and ex-requisites.
 */
public interface InternalDependency extends Describable {
    /**
     * Gets the type.
     * 
     * See the DependencyType section for an explanation of the semantics of each of the possible dependency types. 
     * @return A type which can be pre-req, co-req, or ex-req.
     */
    String getType();
    
    /**
     * Gets the content reference.
     * 
     * The contentElementRef value is the id of the content element that is depended on.<br><br>
     * The value MUST reference the id of a content element.
     * 
     * @return A reference to the content element which is depended on.
     */
    String getContentReference();
    
    /**
     * Gets the content reference operations.
     * 
     * When the depended-on content element is an atomic content element, the operation defined here effectively identifies the 
     * artifact that must be processed for a pre-requisite or co-requisite or not processed for an ex-requisite.<br><br>
     * When the depended-on content element is a CompositeUnit, the operation defined in contentElementRefOperation MUST be the top 
     * level operation defined by the containing CompositeInstallable.
     * The dependency is on application of this operation to the content element identified in contentRef.
     * 
     * @return A collection of strings that represent the value in contentElementRefOperation.
     */
    Collection<String> getContentReferenceOperations();
    
    /**
     * Sets the type.
     * 
     * See the DependencyType section for an explanation of the semantics of each of the possible dependency types.
     *  
     * @param type Type can be pre-req, co-req, or ex-req.
     * <ul>**default value=pre-req</ul>
     */
    void setType(String type);
    
    /**
     * Sets the content reference.
     * 
     * The contentElementRef value is the id of the content element that is depended on.<br><br>
     * The value MUST reference the id of a content element.
     * 
     * @param ref A reference to the content element which is depended on.
     * @throws InvalidInputException if the parameter is null
     */
    void setContentReference(String ref);
    
    /**
     * Sets the content reference operations.
     * 
     * When the depended-on content element is an atomic content element, the operation defined here effectively identifies the 
     * artifact that must be processed for a pre-requisite or co-requisite or not processed for an ex-requisite.<br><br>
     * When the depended-on content element is a CompositeUnit, the operation defined in contentElementRefOperation MUST be the top 
     * level operation defined by the containing CompositeInstallable.
     * The dependency is on application of this operation to the content element identified in contentRef.
     * 
     * @param operations A collection of strings that represent the value in contentElementRefOperation.
     */
    void setContentReferenceOperations(Collection<String> operations);
     
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();
}