/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.operations;

import org.eclipse.cosmos.rm.internal.repository.RepositoryMessages;
import org.eclipse.cosmos.rm.internal.repository.core.FileSystemSMLRepository;
import org.eclipse.cosmos.rm.internal.repository.core.IFileSystemSMLProperties;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.provisional.repository.operations.IProgressListener;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLImportOperation;

/**
 * The implementation of the import operation.  This operation accepts the following
 * arguments in the order listed below:
 * <ol>
 * 	<li>A {@link String} indicating the path of the SML-IF document being imported (required) </li>
 * 	<li>A {@link String} indicating the path of the target directory (required) </li>
 * 	<li>A {@link Boolean} set to true to overwrite files without any notification (optional - 
 * 		false by default) </li>
 * 	<li>A {@link Boolean} set to true to create any structure needed as part of the import 
 * 		operation or false not to create the structure (optional - true by default) </li>
 * </ol>
 * 
 * @author Ali Mehregani
 */
public class FileImportOperation extends AbstractSMLOperation implements ISMLImportOperation
{
	/**
	 * Constructor
	 */
	public FileImportOperation(ISMLRepository repository)
	{
		super(repository);
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.operations.ISMLOperation#getId()
	 */
	public String getId()
	{
		return ISMLImportOperation.ID;
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.operations.ISMLOperation#run()
	 */
	public void run() throws RepositoryOperationException
	{
		boolean overwriteProperty = getRepository().getProperty(IFileSystemSMLProperties.OVERWRITE_WITHOUT_NOTIFICATION, false);
		boolean createStructure = getRepository().getProperty(IFileSystemSMLProperties.CREATE_STRUCTURE, false);
		IProgressListener[] listeners = getProgressListeners();
		try
		{
			// The first argument is expected to be the path to the SML-IF document and
			// the second argument is expected to be the target directory 
			Object[] arguments = getArguments();
			if  (arguments.length < 2 || !super.checkType(arguments[0].getClass(), String.class) || 
				!super.checkType(arguments[1].getClass(), String.class) || 
				(arguments.length >= 3 && !super.checkType(arguments[2].getClass(), Boolean.class)))
				throw new RepositoryOperationException(RepositoryMessages.operationBadArguments);
			
			setProperty (arguments, 2, IFileSystemSMLProperties.OVERWRITE_WITHOUT_NOTIFICATION);
			setProperty (arguments, 3, IFileSystemSMLProperties.CREATE_STRUCTURE);
			
			// Register the progress listeners			
			FileSystemSMLRepository repository = (FileSystemSMLRepository)getRepository(); 
			if (listeners != null)
			{
				for (int i = 0; i < listeners.length; i++)
				{
					repository.addProgressListener(listeners[i]);			
				}
			}
						
			ImportOperation importOperation = new ImportOperation((FileSystemSMLRepository)getRepository());
			importOperation.importFromFileSystemFile((String)arguments[0], (String)arguments[1]);

		}
		finally
		{			
			// Reset the repository properties
			getRepository().setProperty(IFileSystemSMLProperties.OVERWRITE_WITHOUT_NOTIFICATION, overwriteProperty);
			getRepository().setProperty(IFileSystemSMLProperties.CREATE_STRUCTURE, createStructure);
			
			// Remove the progress listeners
			FileSystemSMLRepository repository = (FileSystemSMLRepository)getRepository(); 
			if (listeners != null)
			{
				for (int i = 0; i < listeners.length; i++)
				{
					repository.removeProgressListener(listeners[i]);			
				}
			}
		}
	}

	private void setProperty(Object[] arguments, int argumentInx, String propertyName)
	{
		if (arguments.length > argumentInx)
		{
			if (arguments[argumentInx] instanceof String)
			{
				getRepository().setProperty(propertyName, (String)arguments[argumentInx]);	
			}
			else if (arguments[argumentInx] instanceof Boolean)
			{
				 getRepository().setProperty(propertyName, ((Boolean)arguments[argumentInx]).booleanValue());
			}
		}			
	}
}
