/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Describable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * ConsumptionConstraintType provides the type definition of the Consumption elements of RequirementResourceConstraintType. These 
 * elements are used to express a requirement on the available quantity of a particular resource property such as disk space on a file 
 * system. ConsumptionConstraints represent exclusive use of the defined quantity of the resource property. In other words, consumption 
 * constraints are additive, with each consumption constraint specified in the SDD adding to the total requirement for the specified 
 * resource(s). A consumption constraint is assumed to alter the available quantity such that the portion of the property used to 
 * satisfy one constraint is not available to satisfy another consumption constraint on the same property.
 * <ul>For example, suppose that the target file system has 80 megabytes available. The application of a content elements 
 * InstallArtifact results in installation of files that use 5 megabytes of file space. The application of a second InstallArtifact 
 * results in installation of files that use 2 megabytes of file space. Consumption constraints are additive, so the total space used 
 * for this content element is 7 megabytes, leaving 73 (807) megabytes available on the target file system.</ul>
 */
public interface ConsumptionConstraint extends Describable {
	/**
	 * Gets the property name.
	 * 
	 * The property name can be used to find the property value in the deployment environment. This name may be specified in profiles.
	 * 
	 * @return Names the resource property to test.
	 */
	String getPropertyName();
	
	/**
	 * Gets the value.
	 * 
	 * The result of evaluating this variable expression represents the minimum quantity of the named resource property that MUST be 
	 * available for successful deployment of the defining content elements artifacts. This quantity will be consumed by application 
	 * of the associated artifact.
	 * 
	 * @return A variable expression defining the minimum available 
	 */
	String getValue();
	
	/**
	 * Sets the property name.
	 * 
	 * The property name can be used to find the property value in the deployment environment. This name may be specified in profiles.
	 * 
	 * @param propertyName Names the resource property to test.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setPropertyName(String propertyName);

	/**
	 * Sets the value.
	 * 
	 * The result of evaluating this variable expression represents the minimum quantity of the named resource property that MUST be 
	 * available for successful deployment of the defining content elements artifacts. This quantity will be consumed by application 
	 * of the associated artifact.
	 * 
	 * @param value A variable expression defining the minimum available 
	 * @throws InvalidInputException if the parameter is null
	 */
	void setValue(String value);
	
	/**
	 * Gets the unit of measure.
	 * 
	 * Values for unit SHOULD be well-known units of measure from International System of Units [UNIT]. A unit of measure SHOULD be 
	 * specified for all properties which are measured in any kind of unit.
	 * 
	 * @return Unit of measure used to interpret the consumption value.
	 */
	String getUnitOfMeasure();
	
	/**
	 * Sets the unit of measure.
	 * 
	 * Values for unit SHOULD be well-known units of measure from International System of Units [UNIT]. A unit of measure SHOULD be 
	 * specified for all properties which are measured in any kind of unit.
	 * 
	 * @param unitOfMeasure Unit of measure used to interpret the consumption value.
	 */
	void setUnitOfMeasure(String unitOfMeasure);
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();	 
}