/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Properties;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.util.ConversionUtil;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Translatable;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public abstract class SPIDataObject extends SPIObject implements Translatable {
    private Node node = null;;
    private String sddSchemaType = null;

    public SPIDataObject(Node element, SPISessionImpl session, String sddSchemaType) {
        super(session);
        this.node = element;
        this.sddSchemaType = sddSchemaType;
    }

    public Element getElement() {
        if (node instanceof Element) {
            return (Element) node;
        }
        else {
            return null;
        }
    }
    
    public String getAttribute(String name) {
    	if (node instanceof Element) {
    		return isSetAttribute(name) ? ((Element)node).getAttribute(name) : null;	
    	}
    	else {
    		return null;
    	}    	
    }
    
    public void setAttribute(String name, String value) {
    	if (node instanceof Element) {
    		if (value == null || value.length() == 0) {
    			((Element)node).removeAttribute(name);
    		}
    		else {
    			((Element)node).setAttribute(name, value);
    		}
    	}
    }

    public void unSetElement(String name) {
    	removeContent(getChildren(name, false));
    }

    public void unSetAttribute(String name) {
    	getElement().removeAttribute(name);
    }

    public Collection<Element> getChildren(String name) {
        return getChildren(name, session.isUseCopy());
    }
    
    public Collection<Element> getChildren(String name, boolean copy) {
        Collection<Node> list = getNodeList(name);
        
        Collection<Element> collection = new ArrayList<Element>();

        for (Node node : list) {
            if (node instanceof Element) {
                if (copy) {
                    collection.add((Element) node.cloneNode(true));
                }
                else {
                    collection.add((Element) node);
                }
            }
        }

        return collection;
    }

    /**
     * @param name
     * @return First child in the list.
     */
    public Element getChild(String name) {
        return getChild(name, session.isUseCopy());
    }
    
    public Element getChild(String name, boolean copy) {
        Collection<Node> list = getNodeList(name);

        Element element = null;

        for (Node node : list) {
            if (node instanceof Element) {
                element = (Element) node;

                if (copy) {
                    element = (Element) node.cloneNode(true);
                }
                else {
                    element = (Element) node;
                }

                break;
            }
        }

        return element;
    }
    
    private Collection<Node> getNodeList(String nodeName) {
        Collection<Node> list = new ArrayList<Node>();
        
        NodeList nodeList = node.getChildNodes();

        if (nodeList != null) {
            for (int i = 0; i < nodeList.getLength(); i++) {
                Node node = nodeList.item(i);
                
                if ((node.getLocalName() != null && (node.getLocalName().equals(nodeName) ||
                		nodeName.endsWith(":" + node.getLocalName()))) || node.getNodeName().equals(nodeName) ||
                		node.getNodeName().endsWith(":" + nodeName)) {
                    list.add(node);
                }
            }
        }
        
        return list;
    }

    /**
     * @param name
     * @return True if the node is set. False if otherwise.
     */
    public boolean isSetElement(String name) {
        return getChild(name) != null ? true : false;
    }

    /**
     * @param name
     * @return True if the attribute is set. False if otherwise.
     */
    public boolean isSetAttribute(String name) {
        boolean value;
        String attribute = "";

        if (node instanceof Element) {
            attribute = ((Element) node).getAttribute(name);
        }
        else if (node instanceof Document) {
            NamedNodeMap map = ((Document) node).getAttributes();
            if (map != null) {
                Node n = map.getNamedItem(name);
                if (n != null) {
                    attribute = n.getTextContent();
                }
            }
        }

        if (attribute.length() > 0) {
            value = true;
        }
        else {
            value = false;
        }

        return value;
    }

    public void setContent(SPIDataObject spiObject) {
        setContent(spiObject, session.isUseCopy());
    }
    
    public void setContent(SPIDataObject spiObject, boolean copy) {
        session.testParameter(spiObject, 1, null);
        
        if (spiObject != null) {
            setContent(spiObject.getElement(), copy);
        }
    }

    public void setContent(Element element) {
        setContent(element, session.isUseCopy());
    }
    
    public void setContent(Element element, boolean copy) {
        session.testParameter(element, 1, null);
        
        if (element != null) {
            String elementName;
            
            if (element.getLocalName() != null) {
                elementName = element.getLocalName();
            }
            else {
                elementName = element.getNodeName();
            }
            
            Collection<Element> children = getChildren(elementName, false);
            removeContent(children);
            
            if (copy) {
            	session.insertNode(this.node, element.cloneNode(true), sddSchemaType);
            }
            else {
            	session.insertNode(this.node, element, sddSchemaType);
            }
        }
    }
    
    public <T> void addContent(Collection<T> elements) {
        addContent(elements, session.isUseCopy());
    }
    
    public <T> void addContent(Collection<T> elements, boolean copy) {
        for (T element : elements) {
            if (element instanceof Element) {
            	if (copy) {
            		session.insertNode(this.node, ((Element)element).cloneNode(true), sddSchemaType);
            	}
            	else {
            		session.insertNode(this.node, (Element)element, sddSchemaType);
            	}
            }
            else if (element instanceof SPIDataObject) {
            	if (copy) {
            		session.insertNode(this.node, ((SPIDataObject)element).getElement().cloneNode(true), sddSchemaType);
            	}
            	else {
            		session.insertNode(this.node, ((SPIDataObject)element).getElement(), sddSchemaType);
            	}
            }
        }
    }

    public <T> void removeContent(Collection<T> elements) {
        for (T element : elements) {
            if (element instanceof  Element) {
                this.node.removeChild((Element)element);
            }
            else if (element instanceof SPIDataObject) {
                this.node.removeChild(((SPIDataObject)element).getElement());
            }
        }
    }
    
    public Properties getTranslatableStrings() {
        Properties p = ConversionUtil.dumpStrings(node);
        return p;
    }
}