/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi;

import java.util.Date;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.w3c.dom.Document;

/**
 * Objects implementing this interface contain information that describes a
 * descriptor.  Note that the descriptor may be either a deployment descriptor
 * or a package descriptor.  The set of descriptive information that may be
 * available is the same for either type. 
 *
 */
public interface Descriptor extends Translatable {
	/**
	 * Returns the descriptor's schema version as a String.  The String is in
	 * <dfn>major.minor</dfn> version format.
	 *  
	 * @return The descriptor complies with this version of the Solution Deployment Descriptor Specification
	 */
	String getSchemaVersion();
	
	/**
	 * Returns the descriptor ID.  The ID is in the form of a UUID (Universally Unique
	 * IDentifier), which is a 16 byte number
	 * 
	 * References:
	 * 	http://www.w3.org/TR/xmlschema-2/#hexBinary
	 * 	http://www.ietf.org/rfc/rfc4122.txt
	 * 
	 * 
	 * Note: The java.util.UUID class was introduced in Java 1.5.  To preserve Java 1.4
	 * compatibility, that class is not used here.
	 * 
	 * @return The descriptor ID (UUID)
	 */
	byte[] getDescriptorID();
	
	
	/**
	 * Returns the date of the descriptor's most recent modification.  The returned value 
	 * is the String representation of a corresponding dateTime object
	 * 
	 * Reference:
	 * 	http://www.w3.org/TR/xmlschema-2/#dateTime
	 * 
	 * 
	 * @return Date of the descriptor's most recent modification
	 */
	Date getLastModified();
		
	/**
	 * Sets the descriptor's schema version as a String.  The String is in
	 * <dfn>major.minor</dfn> version format.
	 * 
	 * @param The descriptor complies with this version of the Solution Deployment Descriptor Specification
	 * @throws InvalidInputException if the parameter is null
	 */
	void setSchemaVersion(String schemaVersion) throws InvalidInputException;
	
	
	/**
	 * Sets the descriptor ID.  The ID is in the form of a UUID (Universally Unique
	 * IDentifier), which is a 16 byte number.
	 * 
	 * References:
	 * 	http://www.w3.org/TR/xmlschema-2/#hexBinary
	 * 	http://www.ietf.org/rfc/rfc4122.txt
	 * 
	 * @param descriptorID
	 * @throws InvalidInputException
	 */
	void setDescriptorID(byte[] descriptorID) throws InvalidInputException;
	
	/**
	 * @param lastModified
	 * @throws InvalidInputException
	 */
	void setLastModified(Date lastModified) throws InvalidInputException;
	
	/**
	 * Get the Document that contains the Descriptor
	 * @return Document containing the Descriptor
	 */
	Document getOwnerDocument();
}
