/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.reference;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;

import org.eclipse.cosmos.rm.internal.validation.artifacts.DOMStructure;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationOutput;
import org.eclipse.cosmos.rm.internal.validation.common.SMLIFIdentity;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil.RemoteRetrievalException;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DocumentDOMBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.IdentityDataBuilder;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Node;


/**
 * Represents an URI as specified by the grammar: 
 * {@link http://www.ietf.org/rfc/rfc3986.txt}
 * 
 * @author Ali Mehregani
 * @author John Arwe
 * @author David Whiteman
 */
public class URIReference implements IReferenceExpression
{	
	/**
	 * Reserved URI characters
	 */
	private static final char[] RESERVED_CHARACTERS = new char[] {
		'$', '&', '+', ',', '/',
		':', ';', '=', '?', '@',
		'-', '_', '.', '!', '*',
		'(', ')', '#', '[', ']', 
		'%', '\'', '\\'};
	
	/**
	 * The base URI
	 */
	private URI base;
	
	/**
	 * The base URI infoset property value (may be a relative reference, or null if no syntactically valid xml:base was observed)
	 * in effect on the reference string's containing element (either specified or inherited).  Note that the
	 * reference string's containing element (e.g. sml:uri) may be different from the context element (e.g. the
	 * SML reference element, having sml:ref=true), and therefore it might have a different xml:base value. 
	 */
	private URI baseURIValue;	
	
	/**
	 * The context node is the document node containing
	 * the reference
	 */
	private Node context;
	
	/**
	 * The reference URI.
	 */
	private URI reference;
	
	/**
	 * The document reference
	 */
	private String documentReference;

	/**
	 * Indicates whether this reference has already been
	 * transformed.
	 */
	private boolean transformed;
	
	
	/**
	 * DOM representation of the instance documents
	 */
	private DOMStructure domStructure;
	
	/**
	 * Constructor.  The reference must be a valid XPointer expression.
	 * 
	 * @param context The context node is the root element of the document
	 * containing the reference element
	 * @param reference The string representing the XPointer expression 
	 * @throws URISyntaxException If the URI has an incorrect syntax or the reference is null
	 */
	public URIReference(Node context, String reference) throws URISyntaxException {
		this(context, reference, null);
	}


	/**
	 * Constructor.  The reference must be a valid XPointer expression.
	 * 
	 * @param context The context node is the root element of the document
	 * containing the reference element, i.e. the SML reference element
	 * @param reference The string representing the XPointer expression 
	 * @param base URI value The base URI for the reference, or null if no base URI is available (e.g. for a model base URI value)
	 * @throws URISyntaxException If the URI has an incorrect syntax or the reference is null
	 */
	public URIReference(Node context, String reference, URI baseURIValue ) throws URISyntaxException
	{		
		if (reference == null) { // e.g. when no sml:uri element was found for an SML reference
			throw new URISyntaxException("", SMLValidationMessages.errorReferenceNullURI);
		}
		
		this.context = context;
		
		// Trim the white space and take out line separators
		reference = trim(reference);
		
		// Encode the unsupported US-ASCII characters
		reference = encodeCharacters (reference);
		
		this.reference = new URI(reference);
		verifyImplementationSupportsURI(this.reference);
		
		this.baseURIValue = baseURIValue;
		verifyImplementationSupportsURI(this.baseURIValue);
	}


	/**
	 * @param verifyURI The input URI to verify this implementation fully supports.  May be null.
	 * @throws URISyntaxException If the URI is opaque according to RFC 2396.
	 */
	private void verifyImplementationSupportsURI(URI verifyURI) throws URISyntaxException 
	{
		//	This class's implementation of RFC 3986 URI transformation from relative reference to absolute URI
		//	is fundamentally dependent on 3986's parsing rules.  java.net.URI's current implementation is based on 
		//	RFC 2396 and the two are not entirely compatible; see for example http://java.sun.com/javase/6/jcp/mr2/#java.net
		//	where RFC 3986 support was backed out of Java SE 6 beta 1.
		//
		//	One known issue this test flags is urn-style base URIs.  
		//	java.net parses urn:foo as a relative path
		//	which mergePath happily processes, but java.net.URI throws an exception when the merged (still relative, in its eyes)
		//	path component is used to construct a URI with a non-null scheme (which is absolute, definitionally).
		if (verifyURI != null && verifyURI.isOpaque())
		{
			throw new URISyntaxException("", SMLValidationMessages.errorReferenceOpaqueURI);
		}
	}


	private String encodeCharacters(String uri)
	{
		if (uri == null) {
			return null;
		}
		char[] uriChars = uri.toCharArray();
		StringBuffer buffer = new StringBuffer();
		for (int i = 0; i < uriChars.length; i++)
		{
			
			if (Character.isLetterOrDigit(uriChars[i])		||		// [0-9][A-Z][a-z]
				isCharacterReserved(uriChars[i]))					// [$&+,/:;=?@-_.!*'()#[]]
			{
				buffer.append(uriChars[i]);
				continue;
			}
			
			// Otherwise convert to hex and add to the buffer
			// The hex is expected to be 2 digit max
			buffer.append("%");
			buffer.append(convertToHex((uriChars[i])/16));			
			buffer.append(convertToHex((uriChars[i])%16));
		}
		
		return buffer.toString();
	}

	
	private char convertToHex(int dec)
	{
		return 	dec >= 0 && dec <= 9 ? String.valueOf(dec).toCharArray()[0] :		// [0-9]
				dec <= 15 ? (char)(65 + (dec - 10)) :								// [A-F]
				(char)dec; 															// Should never happen 	
	}

	private int convertToDecimal(char ch)
	{
		return 	Character.isDigit(ch) ? Integer.parseInt(String.valueOf(ch)) :		// [0-9]
				ch >= 65 && ch <= 70 ? 10 + (ch - 65) : 							// [A-F]
				-1;																	// Not a hex digit
	}
	
	private String decodeCharacters(String uri) {
		if (uri == null) {
			return null;
		}
		
		StringBuffer buffer = new StringBuffer();
		int cursorInx = 0;
		int precentInx;
		int uriPartLength = uri.length();
		while (uriPartLength > cursorInx && (precentInx = uri.indexOf('%', cursorInx)) >= 0)
		{
			buffer.append(uri.substring(cursorInx, precentInx));
			if (uriPartLength > precentInx + 2)
			{				
				char firstDigit = uri.charAt(precentInx + 1);
				char secondDigit = uri.charAt(precentInx + 2);
				
				int firstDecimalDigit = convertToDecimal(firstDigit);
				int secondDecimalDigit = convertToDecimal(secondDigit);
				
				// Convert if this is a number
				if (firstDecimalDigit >= 0 && secondDecimalDigit >= 0)
				{
					buffer.append((char)(firstDecimalDigit*16 + secondDecimalDigit));
				}
				// Otherwise just append blindly 
				else
				{
					buffer.append('%' + firstDigit + secondDigit);
				}
			}
			cursorInx = precentInx + 3;
		}
		
		if (uriPartLength > cursorInx) {
			buffer.append(uri.substring(cursorInx));
		}
		return buffer.toString();
	}


	private boolean isCharacterReserved(char c)
	{
		for (int i = 0; i < RESERVED_CHARACTERS.length; i++) {
			if (c == RESERVED_CHARACTERS[i]) {
				return true;
			}
		}
		
		return false;
	}


	private String trim(String str)
	{
		str = str.trim();
		str = str.replaceAll("\\n", "");
		str = str.replaceAll("\\r", "");
		return str;
	}
	
	
	/**
	 * The document reference is basically the scheme+authority+path components 
	 * of the URI.
	 * 
	 * @return the absolute URI portion of the reference
	 * @throws URISyntaxException if the base URI is invalid
	 */
	public String getDocumentReference() throws URISyntaxException
	{		
		if (documentReference == null)
		{			
			String referenceStr = null;
			
			try {
				referenceStr = URLDecoder.decode(reference.toString(), "UTF-8");
			} catch (UnsupportedEncodingException e) {
				e.printStackTrace();
				throw new URISyntaxException(reference.toString(), e.getMessage());
			}
			
			int fragmentInx = referenceStr == null ? -1 : referenceStr.indexOf('#');
			if (fragmentInx >= 0) {				
				referenceStr = referenceStr.substring(0, fragmentInx).trim();				
			}
			documentReference = referenceStr;
			
			// Transform the URI if the aliases cannot be found
			//	Note: must call transform even if the ref string is empty (==""), 
			//				since that case is a non-empty path + null fragment i.e. a relative reference
			if (!isDocumentEmbedded(referenceStr) && 
				!isTransformed())
			{
				documentReference = null;
				transform();
				return getDocumentReference();
			}
		}
		
		return decodeCharacters(documentReference);
	}


	/**
	 * Returns the fragment component of the URI
	 * 
	 * @return The fragment component of the reference, or null if no fragment exists
	 * @throws URISyntaxException 
	 */
	public String getFragment() throws URISyntaxException
	{
		/* This has the side effect of transforming the URI if needed */
		getDocumentReference();		
		return reference == null ? null : decodeCharacters(reference.getFragment());
	}
	
	private boolean isDocumentEmbedded(String document) {
		if (domStructure == null) {
			domStructure = (DOMStructure) SMLValidatorUtil.retrieveDataStructure(DocumentDOMBuilder.ID);
		}

		return domStructure != null && domStructure.get(document) != null;
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.reference.IReferenceExpression#isTransformed()
	 */
	public boolean isTransformed() {
		return transformed;
	}


	/**
	 * Checks to see if transform() is likely to be successful, at least with respect to the base URI value available.
	 * 
	 * @return message if the base URI has a problem, null if the base URI appears to be OK or is unneeded
	 */
	public String checkIfBaseURIReadyToTransform() {
		return checkIfBaseURIReadyToTransform(false);
	}
	
	/**
	 * Checks to see if transform() is likely to be successful, at least with respect to the base URI value available.
	 * 
	 * @return message if the base URI has a problem, null if the base URI appears to be OK or is unneeded
	 */
	private String checkIfBaseURIReadyToTransform(Boolean updateBaseURI) {
		// DLW - This method should be changed to return a new object that holds a boolean
		// indicating success/fail, plus the message.  Better design than checking for null
		// to see if successful.
		
		String message = null;

		// A transformation (and hence a base URI) is not required if the URI is either absolute or only
		// contains a fragment
		if (reference.isAbsolute() || isFragmentOnly()) {
			// Its current value is fine
		}

		//	Figure out the correct base URI to use:
		//	- If the input reference is absolute, no "external" base URI is needed.
		//	- If the input reference is a same-document reference (fragment only), no base URI is needed.
		//	- If an xml:base specification exists in the ancestor axis, and it is absolute, use the nearest one.
		//	- If the nearest xml:base specification exists in the ancestor axis, but it is a relative reference (after recursively searching for an absolute base URI), the document is invalid.
		//	- If an smlif:baseURI specification exists on the document, use it.
		//	- If an smlif:baseURI specification exists on the model, use it.
		//	- Else the document is invalid.
		//	Cannot use context.getBaseURI() since "context" is the wrong element, e.g. context is the SML reference
		//	not the sml:uri element.  An xml:base on the sml:uri would be missed by context.getBaseURI().
		
		else if (baseURIValue == null) {
			// Error: need base URI but have none at all
			message = SMLValidationMessages.errorMissingBaseURI;
		} else if (baseURIValue.isAbsolute()) {
			//	Either xml:base was specified within the document, or a document URI was provided 
			//	(from an ancestor xml:base, or from an smlif:baseURI on the document or model)
			if (updateBaseURI) {
				base = baseURIValue;
			}
		} 
		else 
		{
			// Error: non-empty base URI was provided, an absolute base URI is needed,
			//	but the specified value is not an absolute URI.  The implementation might be able to provide
			//	an absolute URI in the [base URI] infoset property using the SML-IF document's URI as a base, 
			//	but SML-IF does not allow that.
			message = NLS.bind(SMLValidationMessages.errorRelativeBaseURI, baseURIValue.toString());
		}
		
		SMLIFIdentity identity = (SMLIFIdentity)SMLValidatorUtil.retrieveDataStructure(DataBuilderRegistry.TOP_LEVEL_MODE, IdentityDataBuilder.ID);
		if (message != null && identity != null && identity.getTestUtilityRunning())	//	Allow JUnits to continue running in the absence of a full SML-IF document
		{
			String artificial = SMLValidationMessages.artificialIdentity ;
			String baseURI = identity.getBaseURI();
			if (baseURI == null) {
				message = artificial + " " + SMLValidationMessages.errorBadBaseURI ;
			} else {
				try {
					URI tryBase = new URI(baseURI);
					if (!tryBase.isAbsolute()) {
						message = NLS.bind(SMLValidationMessages.errorBadValue, 
								new String [] { SMLValidationMessages.modelBaseURI , baseURI }) ;
						message = artificial + " " + message + SMLValidationMessages.errorAbsoluteURIRequired;
					} else {
						message = null; // Successfully fished something usable
										// out of the scaffolding
						if (updateBaseURI) {
							base = tryBase;
						}
					}
				}
				catch (URISyntaxException e) {
					message = e.getMessage();
				}
			}
		}

		return message;
	}
	
	/**
	 * The transformation will be based on [RFC 3986]: {@link http://www.ietf.org/rfc/rfc3986.txt}.
	 * See section 5 for more details.
	 * 
	 * @throws URISyntaxException If the base URI is invalid 
	 * @see org.eclipse.cosmos.rm.internal.validation.reference.IReferenceExpression#transform()
	 */
	public void transform() throws URISyntaxException
	{		
		transformed = true;

		//	Figure out the correct base URI to use:
		//	- If the input reference is absolute, no "external" base URI is needed.
		//	- If the input reference is a same-document reference (fragment only), no base URI is needed.
		//	- If an xml:base specification exists in the ancestor axis, and it is absolute, use the nearest one.
		//	- If the nearest xml:base specification exists in the ancestor axis, but it is a relative reference (after recursively searching for an absolute base URI), the document is invalid.
		//	- If an smlif:baseURI specification exists on the document, use it.
		//	- If an smlif:baseURI specification exists on the model, use it.
		//	- Else the document is invalid.
		//	Cannot use context.getBaseURI() since "context" is the wrong element, e.g. context is the SML reference
		//	not the sml:uri element.  An xml:base on the sml:uri would be missed by context.getBaseURI().
		
		// A transformation is not required if the URI is either absolute or only
		// contains a fragment
		if (reference.isAbsolute() || isFragmentOnly()) {
			return;
		}

	
		String message = checkIfBaseURIReadyToTransform(true);	//	Not only check, but update the base URI if necessary
		if (message != null) {
			throw new URISyntaxException(IValidationConstants.EMPTY_STRING, message);
		}
		
		String scheme, authority, path, query, fragment;
		
		// Determine the base URI
		if (defined(reference.getScheme())) {
			scheme = reference.getScheme();
			authority = reference.getAuthority();
			path = reference.getPath();
			query = reference.getQuery();
		} else {
			if (defined(reference.getAuthority())) {
				authority = reference.getAuthority();
				path = reference.getPath();
				query = reference.getQuery();
			} else {
				if (defined(reference.getPath())) {
					path = reference.getPath().startsWith(ISMLConstants.FORWARD_SLASH) ? 
							removeDotSegments(reference.getPath()) : removeDotSegments(mergePath(base, reference));
					query = reference.getQuery();
				} else {
					path = base.getPath();
					query = defined(reference.getQuery()) ? reference.getQuery() : base.getQuery();					
				}
				authority = base.getAuthority();				
			}
			scheme = base.getScheme();
		}
		fragment = reference.getFragment();
		
		reference = new URI(decodeCharacters(scheme), 
				decodeCharacters(authority), 
				decodeCharacters(path), 
				decodeCharacters(query), 
				decodeCharacters(fragment));
	}


	private boolean isFragmentOnly()
	{		
		return 	isNull(reference.getAuthority()) && isNull(reference.getHost()) && isNull(reference.getPath()) &&
				isNull(reference.getQuery()) && !isNull(reference.getFragment());
	}

	
	private boolean isNull(String str)
	{
		return str == null || str.length() <= 0;
	}

	/**
	 * See page 32 of {@link http://www.ietf.org/rfc/rfc3986.txt}, e.g. {@link http://tools.ietf.org/html/rfc3986#section-5.2.3}
	 * 
	 * @param base The base URI
	 * @param relative The relative uri
	 * @return Merged path of the base and the relative URI
	 */
	private String mergePath(URI base, URI relative)
	{	// Caller is responsible for ensuring that relative.getPath != null, else will get NPE in here
		if (defined(base.getAuthority()) && !defined(base.getPath()))
		{
			return ISMLConstants.FORWARD_SLASH + relative.getPath();
		}
		else
		{
			String basePath = base.getPath();
			if (!defined(basePath))
			{
				basePath = IValidationConstants.EMPTY_STRING;	
			}
			else 
			{
				int rightMostSlash = basePath.lastIndexOf(ISMLConstants.FORWARD_SLASH);
				//	RFC 3986 5.2.3: ...excluding any characters after the right-most "/" - note AFTER not INCLUDING
				//	The / is part of the path value.  / delimits path segments -within- a path but is still part of the path.
				//	The empty string is a valid path segment according to the 3986 ABNF.
				basePath = rightMostSlash >= 0 ? basePath.substring(0, rightMostSlash + 1) : IValidationConstants.EMPTY_STRING;
			}
			return basePath + relative.getPath();
		}		
	}


	/**
	 * See page 33 of {@link http://www.ietf.org/rfc/rfc3986.txt} 
	 * 
	 * @param input The input path
	 * @return removes the dot segments based on the RFC above
	 */
	private String removeDotSegments(String input)
	{
		String output = "";
		String matchedPrefix;
		
		while (input.length() > 0)
		{
			/* If the input buffer begins with a prefix of "../" or "./",
	           then remove that prefix from the input buffer; otherwise*/
			if ((matchedPrefix = findMatch(input, new String[]{"../", "./"}, new String[0])) != null) 
			{
				int length = matchedPrefix.length();
				input = input.length() > length ? input.substring(length) : IValidationConstants.EMPTY_STRING;
			}
			
			/* if the input buffer begins with a prefix of "/./" or "/.",
	           where "." is a complete path segment, then replace that
	           prefix with "/" in the input buffer; otherwise, */
			else if ((matchedPrefix = findMatch(input, new String[]{"/./"}, new String[]{"/."})) != null) 	
			{
				int length = matchedPrefix.length();
				input = input.length() > length ? input.substring(length) : IValidationConstants.EMPTY_STRING;
				input = ISMLConstants.FORWARD_SLASH + input;
			}

			
			/* if the input buffer begins with a prefix of "/../" or "/..",
			   where ".." is a complete path segment, then replace that
           	   prefix with "/" in the input buffer and remove the last
           	   segment and its preceding "/" (if any) from the output
           	   buffer; otherwise, */
			else if ((matchedPrefix = findMatch(input, new String[]{"/../"}, new String[]{"/.."})) != null) 	
			{
				int length = matchedPrefix.length();
				input = input.length() > length ? input.substring(length) : IValidationConstants.EMPTY_STRING;
				input = ISMLConstants.FORWARD_SLASH + input;
				
				int inx = output.lastIndexOf(ISMLConstants.FORWARD_SLASH);
				output = inx > 0 ? output.substring(0, inx) : IValidationConstants.EMPTY_STRING;  				
			}
			
			/* if the input buffer consists only of "." or "..", then remove
           	   that from the input buffer; otherwise, */
			else if ((matchedPrefix = findMatch(input, new String[0], new String[]{".", ".."})) != null)
			{
				input = IValidationConstants.EMPTY_STRING;				
			}
			
			/* move the first path segment in the input buffer to the end of
           	   the output buffer, including the initial "/" character (if
           	   any) and any subsequent characters up to, but not including,
           	   the next "/" character or the end of the input buffer. */
			else
			{
				int inx = input.indexOf(ISMLConstants.FORWARD_SLASH);
				
				if (inx == 0)	//	starts with slash: copy the slash and continue with the rest
				{
					output += ISMLConstants.FORWARD_SLASH;
					input = input.substring(1);
					inx = input.indexOf(ISMLConstants.FORWARD_SLASH);
				}
					
				if (inx > 0)		//	path segment followed by slash: copy the path segment and continue
				{
					output += input.substring(0, inx);
					input = input.substring(inx);
				}
				else				//	no more slashes => final path segment: copy the rest, done
				{
					output += input;
					input = IValidationConstants.EMPTY_STRING;
				}				 
			}
			
		}
		
		return output;
	}


	private String findMatch(String input, String[] prefix, String[] equalityCheck)
	{
		for (int i = 0; i < prefix.length; i++) {
			if (input.startsWith(prefix[i])) {
				return prefix[i];
			}
		}

		for (int i = 0; i < equalityCheck.length; i++) {
			if (input.equals(equalityCheck[i])) {
				return equalityCheck[i];
			}
		}
		
		return null;
	}


	private boolean defined(String field) {
		return field != null && field.length() > 0;
	}


	/**
	 * Retrieves the document node and returns the result
	 * 
	 * @return The document node
	 * @throws URISyntaxException 
	 * @throws RemoteRetrievalException 
	 */
	public Node retrieveDocumentDOM() throws URISyntaxException {		
		String document = getDocumentReference();
		
		// Return the context node if the document reference is missing
		if (document.length() <= 0) {
			return this.context;
		}
		
		// Return what's stored under aliases
		String scheme = reference.getScheme();
		Node documentNode = document == null ? null : domStructure.get(document);
		if (scheme == null || documentNode != null) {
			return documentNode;
		}
		
		/* Otherwise we'll need to download the reference document */ 
		try {
			return SMLValidatorUtil.retrieveRemoteDocument(reference.toString());
		} catch (RemoteRetrievalException e) {
			return null;
		}
	}

	/**
	 * @return the reference
	 */
	protected URI getReference() {
		return reference;
	}

	/**
	 * @param reference the reference to set
	 */
	protected void setReference(URI reference) {
		this.reference = reference;
	}

	/**
	 * @return the base
	 */
	protected URI getBase() {
		return base;
	}


	/**
	 * @param base the base to set
	 */
	public void setBase(URI base) throws URISyntaxException {
		this.base = base;
		verifyImplementationSupportsURI(this.base);
		this.baseURIValue = this.base;
	}	
	
	/**
	 * Given a potential SML reference value, transform it into a non-relative value if possible
	 * and (optionally) log errors if it is not viable.
	 *
	 * @param candidateRef The potential SML reference
	 * @param baseURI The base URI used to transform the candidate SML reference to a non-relative value, 
	 * 				if transformation is necessary.  May be null if no base URI is available, or may be a relative reference.
	 * @param logger The validation output target, to which error messages are written.  May be null.
	 * @param lineNumber The source line number, used for diagnostic output.  
	 * 				A value of -1 indicates an unknown line number.
	 * @param diagnosticRefName The "type" of the SML reference, e.g. rule or document, used for diagnostic output.  
	 * @return The normalized SML reference value.  If the candidate SML reference is viable, a non-relative with an optional fragment component, else null.
	 */
	public static String validURIReference(String candidateRef, URI baseURI, IValidationOutput<?, ?> logger, int lineNumber, String diagnosticRefName) {
		URIReference candidateRefURI = URIReference.validReference(candidateRef, baseURI, logger, lineNumber, diagnosticRefName);
		if (candidateRefURI == null) {
			return null;
		}

		String returnValue = null;							
		try {
			String fragment = candidateRefURI.getFragment();								
			returnValue = fragment == null ? candidateRefURI.getDocumentReference() : candidateRefURI.getDocumentReference() + "#" + fragment; 								
		}
		catch (URISyntaxException e) {
			if (logger != null) {
				String message = NLS.bind(SMLValidationMessages.errorBadGetDocRefValue, 
						new String [] { diagnosticRefName , candidateRef }) + " " + e.getMessage();
				logger.reportMessage(ValidationMessageFactory.createErrorMessage(
						lineNumber, message ));
			}
			return null;
		}
		return returnValue;
	}

	/**
	 * Given a potential alias (or alias prefix) value, transform it into a viable absolute alias value if possible
	 * and (optionally) log errors if it is not viable.
	 *
	 * @param candidateAlias The potential alias
	 * @param baseURI The base URI used to transform the candidate alias to an absolute URI, 
	 * 				if transformation is necessary.  May be null if no base URI is available, or may be a relative reference.
	 * @param logger The validation output target, to which error messages are written.  May be null.
	 * @param lineNumber The source line number, used for diagnostic output.  
	 * 				A value of -1 indicates an unknown line number.
	 * @param diagnosticAliasName The "type" of the alias, e.g. rule or document, used for diagnostic output.  
	 * @return The normalized alias value.  If the candidate alias is viable, an absolute URI with no fragment component, else null.
	 */
	public static String validAliasOrAliasPrefix(String candidateAlias, URI baseURI, IValidationOutput<?, ?> logger, int lineNumber, String diagnosticAliasName) {
		URIReference candidateAliasURI = URIReference.validAliasOrAliasPrefixAsURIRef(candidateAlias, baseURI, logger, lineNumber, diagnosticAliasName);
		if (candidateAliasURI == null) {
			return null;
		}

		String returnValue = null;							
		try
		{	//	This is VERY unlikely to fail (throw an exception) since the method called above already resulted in 
			//	gDR being invoked, and its implementation caches the result in a local component.
			returnValue = candidateAliasURI.getDocumentReference();								
		} catch (URISyntaxException e) {
			if (logger != null) {
				String message = NLS.bind(SMLValidationMessages.errorBadGetDocRefValue, 
						new String [] { diagnosticAliasName , candidateAlias }) + " " + e.getMessage();
				logger.reportMessage(ValidationMessageFactory.createErrorMessage(
						lineNumber, message ));
			}
			return null;
		}
		return returnValue;
	}

	
	/**
	 * Given a potential alias (or alias prefix) value, transform it into a viable absolute alias value if possible
	 * and (optionally) log errors if it is not viable.
	 *
	 * @param candidateAlias The potential alias
	 * @param baseURI The base URI used to transform the candidate alias to an absolute URI, 
	 * 				if transformation is necessary.  May be null if no base URI is available, or may be a relative reference.
	 * @param logger The validation output target, to which error messages are written.  May be null.
	 * @param lineNumber The source line number, used for diagnostic output.  
	 * 				A value of -1 indicates an unknown line number.
	 * @param diagnosticAliasName The "type" of the alias, e.g. rule or document, used for diagnostic output.  
	 * @return The normalized alias value.  If the candidate alias is viable, an absolute URI with no fragment component, else null.
	 */
	public static URI validAliasOrAliasPrefixAsURI( String candidateAlias , URI baseURI , IValidationOutput<?, ?> logger , int lineNumber , String diagnosticAliasName )
	{
		URIReference candidateAliasURI = URIReference.validAliasOrAliasPrefixAsURIRef(candidateAlias, baseURI, logger, lineNumber, diagnosticAliasName);
		if (candidateAliasURI == null) {
			return null;
		}
		return candidateAliasURI.reference;
	}

	
	/**
	 * Given a potential alias (or alias prefix) value, transform it into a viable absolute alias value if possible
	 * and (optionally) log errors if it is not viable.
	 *
	 * @param candidateAlias The potential alias
	 * @param baseURI The base URI used to transform the candidate alias to an absolute URI, 
	 * 				if transformation is necessary.  May be null if no base URI is available, or may be a relative reference.
	 * @param logger The validation output target, to which error messages are written.  May be null.
	 * @param lineNumber The source line number, used for diagnostic output.  
	 * 				A value of -1 indicates an unknown line number.
	 * @param diagnosticAliasName The "type" of the alias, e.g. rule or document, used for diagnostic output.  
	 * @return The normalized alias value.  If the candidate alias is viable, an absolute URI with no fragment component, else null.
	 */
	public static URIReference validAliasOrAliasPrefixAsURIRef( String candidateAlias , URI baseURI , IValidationOutput<?, ?> logger , int lineNumber , String diagnosticAliasName )
	{
		URIReference candidateAliasURI = URIReference.validAbsoluteURI(candidateAlias, baseURI, logger, lineNumber, diagnosticAliasName);
		if (candidateAliasURI == null) {
			return null;
		}

		// What was returned may be "" if the original input was a same-document reference ("" or fragment-only).  If it is fragment-only, we need to catch that here since the base URI checks would not
		//	have fired in the called methods.  If it has no fragment (it is not fragment-only), then it is simply a relative reference and that
		//	is allowed as long as an appropriate 
		//	Both cases are covered by checking for the presence of a fragment component, since SML-IF prohibits fragments on alias
		if (candidateAliasURI.reference.getFragment() != null) {
			if (logger != null) {
				String message = NLS.bind(SMLValidationMessages.errorAliasHasFragment, 
						new String [] { diagnosticAliasName , candidateAlias });
				logger.reportMessage(ValidationMessageFactory.createErrorMessage(
						lineNumber, message ));
			}
			return null;
		}

		try {
			@SuppressWarnings("unused")
			String transformedURI = candidateAliasURI.getDocumentReference();								
		} catch (URISyntaxException e) {
			if (logger != null) {
				String message = NLS.bind(SMLValidationMessages.errorBadGetDocRefValue, 
						new String [] { diagnosticAliasName , candidateAlias }) + " " + e.getMessage();
				logger.reportMessage(ValidationMessageFactory.createErrorMessage(
						lineNumber, message ));
			}
			return null;
		}
		return candidateAliasURI;
	}

	
	/**
	 * Given a potential model base URI value, test its viability
	 * and (optionally) log errors if it is not viable.
	 *
	 * @param candidateModelBaseURI The potential model base URI
	 * @param baseURI The base URI used to transform the candidate model base URI to an absolute URI, 
	 * 				if transformation is necessary.  May be null if no base URI is available, or may be a relative reference.
	 *                 In practice, this value will be non-null only when an ancestor xml:base specification is in effect.
	 * @param logger The validation output target, to which error messages are written.  May be null.
	 * @param lineNumber The source line number, used for diagnostic output.  
	 * 				A value of -1 indicates an unknown line number.
	 * @return true If the input value is a viable model base URI, i.e. it is an absolute URI with no fragment component
	 */
	public static boolean validModelBaseURI( String candidateModelBaseURI , URI baseURI , IValidationOutput<?, ?> logger , int lineNumber )
	{	 
		return validAliasOrAliasPrefix( candidateModelBaseURI , baseURI , logger , lineNumber , SMLValidationMessages.modelBaseURI ) == null ? false : true;
	}

	/**
	 * Given a potential model base URI value, test its viability
	 * and (optionally) log errors if it is not viable.
	 *
	 * @param candidateModelBaseURIString The potential model base URI
	 * @param baseURI The base URI used to transform the candidate model base URI to an absolute URI, 
	 * 				if transformation is necessary.  May be null if no base URI is available, or may be a relative reference.
	 *                 In practice, this value will be non-null only when an ancestor xml:base specification is in effect.
	 * @param logger The validation output target, to which error messages are written.  May be null.
	 * @param lineNumber The source line number, used for diagnostic output.  
	 * 				A value of -1 indicates an unknown line number.
	 * @return URI If the input value is a viable model base URI, i.e. it is an absolute URI with no fragment component, else null
	 */
	public static URI validModelBaseURIAsURI( String candidateModelBaseURIString , URI baseURI , IValidationOutput<?, ?> logger , int lineNumber )
	{	 
		String modelBaseURIString = validAliasOrAliasPrefix( candidateModelBaseURIString , baseURI , logger , lineNumber , SMLValidationMessages.modelBaseURI ) ;
		if (modelBaseURIString == null)
		{
		return null;	
		}
		URIReference modelBaseURIReference = URIReference.validReference(modelBaseURIString, null, logger, lineNumber, SMLValidationMessages.modelBaseURI);
		if (modelBaseURIReference == null)
		{
			return null;
		}
		return modelBaseURIReference.reference;
	}

	/**
	 * Given a potential absolute URI, check that it is transformable to an absolute URI with no fragment component
	 * and log any errors.
	 *
	 * @param candidateAbsoluteURI The potential absolute URI
	 * @param baseURI The base URI used to transform the candidate alias to an absolute URI, 
	 * 				if transformation is necessary.  May be null if no base URI is available, or may be a relative reference.
	 * @param logger The validation output target, to which error messages are written.  May be null.
	 * @param lineNumber The source line number, used for diagnostic output.  
	 * 				A value of -1 indicates an unknown line number.
	 * @param diagnosticAbsoluteURIName The "type" of the absolute URI, e.g. rule or document, used for diagnostic output.  
	 * @return If the candidate absolute URI is viable, an absolute URI with no fragment component, else null.
	 */
	private static URIReference validAbsoluteURI( String candidateAbsoluteURI , URI baseURI , IValidationOutput<?, ?> logger , int lineNumber , String diagnosticAbsoluteURIName )
	{
		String message = null;	
		URIReference candidateAbsoluteURIReference = URIReference.validReference(candidateAbsoluteURI, baseURI, logger, lineNumber, diagnosticAbsoluteURIName);
		if (candidateAbsoluteURIReference == null) {
			return null;
		}

		// What was returned may be "" if the original input was a same-document reference ("" or fragment-only).  If it is fragment-only, we need to catch that here since the base URI checks would not
		//	have fired in the called methods.  If it has no fragment (it is not fragment-only), then it is simply a relative reference and that
		//	is allowed as long as an appropriate 
		//	Both cases are covered by checking for the presence of a fragment component, since SML-IF prohibits fragments on alias
		if (!candidateAbsoluteURIReference.reference.isAbsolute()) {
			if (logger != null) {
				message = NLS.bind(SMLValidationMessages.errorBadValue, 
						new String [] { diagnosticAbsoluteURIName , candidateAbsoluteURI }) 
						+ SMLValidationMessages.errorAbsoluteURIRequired;
				logger.reportMessage(ValidationMessageFactory.createErrorMessage(
						lineNumber, message ));
			}
			return null;
		}
		return candidateAbsoluteURIReference;
	}

	/**
	 * Given a potential SML reference, check that it is transformable to an absolute URI with an optional fragment component
	 * and log any errors.
	 *
	 * @param candidateRef The potential SML reference
	 * @param baseURI The base URI used to transform the candidate SML reference to an absolute URI with an optional fragment component, 
	 * 				if transformation is necessary.  May be null if no base URI is available, or may be a relative reference.
	 * @param logger The validation output target, to which error messages are written.  May be null.
	 * @param lineNumber The source line number, used for diagnostic output.  
	 * 				A value of -1 indicates an unknown line number.
	 * @param diagnosticRefName The "type" of the SML reference, e.g. rule alias or document alias, used for diagnostic output.  
	 * @return If the candidate SML reference is viable, an absolute URI with an optional fragment component, else null.
	 */
	private static URIReference validReference( String candidateRef , URI baseURI , IValidationOutput<?, ?> logger , int lineNumber , String diagnosticRefName )
	{
		String message = null;	
		URIReference candidateReference = null;
		try {
			candidateReference = new URIReference(null,candidateRef,baseURI);	// exception only if alias value itself bad, base URI not used in check
		} catch (URISyntaxException e) {
			if (logger != null) {
				message = e.getMessage().trim();
				if (message.endsWith(":")) {
					message = message.substring(0, message.length() - 1);
				}	
				message = NLS.bind(SMLValidationMessages.errorBadValue, 
						new String [] { diagnosticRefName , candidateRef }) + message;
				logger.reportMessage(ValidationMessageFactory.createErrorMessage(
						lineNumber, message ));
			}
			return null;
		}
		message = candidateReference.checkIfBaseURIReadyToTransform(true);	//	"true" actually updates the base URI	
		if (message != null) {
			if (logger != null) {
				message = NLS.bind(SMLValidationMessages.errorBadBaseURIValue, 
						new String [] { diagnosticRefName , candidateRef }) + message;
				logger.reportMessage(ValidationMessageFactory.createErrorMessage(
						lineNumber, message ));
			}
			return null;
		}

		if (!candidateReference.isFragmentOnly()) {
			try {
				candidateReference.getDocumentReference();
			} catch (URISyntaxException e) {
				if (logger != null) {
					message = NLS.bind(SMLValidationMessages.errorBadGetDocRefValue, 
							new String [] { diagnosticRefName , candidateRef }) + " " + e.getMessage();
					logger.reportMessage(ValidationMessageFactory.createErrorMessage(
							lineNumber, message ));
				}
				return null;
			}
		}

		return candidateReference;
	}

}
