/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.reference;

import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.NamespaceContext;

import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * This class is used to represent an XPointer expression that can be
 * evaluated.  Instances of this class should be obtained by compiling a
 * string representation of the XPointer expression: {@link XPointer#compile(String)}
 * 
 * @author Ali Mehregani
 */
public class XPointerExpression
{
	/**
	 * The increment used to increase the size of the child sequence array
	 */
	private static final int CHILD_SEQ_INCREMENT = 5;
	
	/**
	 * Indicates a bare name type
	 */
	protected static final byte BARE_NAME_TYPE = 0x00;
	
	/**
	 * Indicates a child sequence type
	 */
	protected static final byte CHILD_SEQUENCE_TYPE = 0x01;
	
	/**
	 * Indicates a full xpointer type
	 */
	protected static final byte FULL_XPOINTER_TYPE = 0x02;
	
	/**
	 * The type of this expression
	 */
	private byte type;
	
	/**
	 * The bare name.  This field is expected to be set if type
	 * is BARE_NAME_TYPE.
	 */
	private String bareName;
	
	/**
	 * Schemes associated with this expression
	 */
	private List<IXScheme> schemes;
	
	/**
	 * Adds a child sequence to this expression
	 */
	private int[] childSequence;
	
	/**
	 * Indicates the current index of the child sequence
	 */
	private int childSequenceIndex;
	
	/**
	 * The visibility of the constructor is limited.
	 * 
	 * @param type The type of this expression
	 * @see #BARE_NAME_TYPE
	 * @see #CHILD_SEQUENCE_TYPE
	 * @see #FULL_XPOINTER_TYPE
	 */
	protected XPointerExpression(byte type)
	{
		this.type = type;
		schemes = new ArrayList<IXScheme>();
		childSequence = new int[CHILD_SEQ_INCREMENT];
		childSequenceIndex = 0;
	}

	
	/**
	 * @return the bareName
	 */
	public String getBareName()
	{
		return bareName;
	}

	/**
	 * @param bareName the bareName to set
	 */
	public void setBareName(String bareName)
	{
		this.bareName = bareName;
	}

	/**
	 * @return the type
	 */
	public byte getType()
	{
		return type;
	}

	/**
	 * Add a scheme to this expression
	 * 
	 * @param scheme The scheme to be added
	 */
	public void addScheme(IXScheme scheme)
	{
		schemes.add(scheme);		
	}


	/**
	 * Add a child sequence to this expression
	 * 
	 * @param child
	 */
	public void addChildSequence(int child)
	{
		/* The size of the child sequence array needs to be increased */
		if (childSequenceIndex >= childSequence.length)
		{			
			int[] newChildSequence = new int[childSequence.length + CHILD_SEQ_INCREMENT];
			System.arraycopy(childSequence, 0, newChildSequence, 0, childSequence.length);
			childSequence = newChildSequence;
		}
		
		childSequence[childSequenceIndex++] = child - 1; /* Subtract 1 to make it 0-indexed */
	}
	
	
	/**
	 * Equivalent to evaluate(null, Object)
	 * @see #evaluate(NamespaceContext, Object)
	 */
	public Object evaluate(Object context) throws BadContextException, BadExpressionException
	{
		return evaluate (null, context);
	}
	
	
	/**
	 * Evaluates this XPointer expression using the context passed in
	 * 
	 * @param context The context to be used to evaluate this expression
	 * 
	 * @param namespaceContext The namespace context to be used when evaluating this
	 * XPointer expression.
	 * @return An object resulting from evaluating this expression
	 * 
	 * @throws BadContextException If the context of the expression is not valid. 
	 * @throws BadExpressionException If the expression of a scheme is invalid
	 */
	public Node evaluate(NamespaceContext namespaceContext, Object context) throws BadContextException, BadExpressionException
	{
		if (!(context instanceof Node))
			throw new BadContextException(SMLValidationMessages.errorBadContext);
		
		if (namespaceContext != null)
		{
			for (int i = 0, schemeCount = schemes.size(); i < schemeCount; i++)
			{
				IXScheme scheme = schemes.get(i);
				scheme.setNamespaceContext(namespaceContext);
			}
		}
		
		Node contextNode = (Node)context;
		
		/* A bare name expression */
		if (BARE_NAME_TYPE == type)
		{
			return retrieveNodeByBareName(contextNode);
		}
		
		/* A child sequence */
		else if (CHILD_SEQUENCE_TYPE == type)
		{
			Node node = getBareName() == null ? (Node)context : (Node)retrieveNodeByBareName(contextNode);
			if (node == null)
				return null;
			
			NodeList childNodes = null;
			int[] childSequence = getChildSequence();
			for (int i = 0; i < childSequence.length; i++)
			{
				childNodes = node == null ? null : node.getChildNodes();
				if (childNodes == null || childSequence[i] < 0)
					return null;				
				
				int currentInx = 0;
				int inxWithoutWhiteSpace = 0;
				String nodeValue = null;
				
				while (inxWithoutWhiteSpace <= childSequence[i])
				{
					if (currentInx < childNodes.getLength())
					{
						node = childNodes.item(currentInx);
					}
					else
					{
						return null;
					}
					
					if (Node.TEXT_NODE != node.getNodeType() || ((nodeValue = node.getNodeValue()) != null && nodeValue.trim().length() > 0))
						inxWithoutWhiteSpace++;
					currentInx ++;
				}				
			}			
			
			return node;
		}
		
		/* A full XPointer expression */
		else if (FULL_XPOINTER_TYPE == type)
		{
			Object ctx = context;
			for (int i = 0, schemeCount = schemes.size(); i < schemeCount; i++)
			{
				IXScheme scheme = (IXScheme)schemes.get(i);
				ctx = scheme.evaluate(ctx);
			}
			
			if (ctx == null)
				return null;
			
			if (!(ctx instanceof NodeList[]))
				throw new BadExpressionException (SMLValidationMessages.errorWrongReturnType);
			NodeList[] nodeListArray = (NodeList[])ctx;
			NodeList nodeList = null;
			if (nodeListArray.length <= 0)
			{
				return null;
			}
			else if (nodeListArray.length > 1 || (nodeList = nodeListArray[0]).getLength() > 1)
			{
				throw new BadExpressionException (SMLValidationMessages.errorWrongReturnScope);
			}
			
			return nodeList.getLength() == 0 ? null : nodeList.item(0);
		}
		
		return null;
	}
	
	private Node retrieveNodeByBareName(Node context)
	{
		Document document = context.getOwnerDocument();
		return document == null ? null : document.getElementById(getBareName());
	}
	
	/**
	 * Returns the schemes associated with this XPointer expression
	 * 
	 * @return The schemes of this expression
	 */
	public List<IXScheme> getSchemes()
	{
		return schemes;
	}
	
	
	/**
	 * Retuns a copy of the child sequence
	 * 
	 * @return The child sequence associated with this expression
	 */
	public int[] getChildSequence()
	{
		int[] childSeq = new int[childSequenceIndex];
		System.arraycopy(childSequence, 0, childSeq, 0, childSeq.length);
		return childSeq;
	}
}
