/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.validation;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.util.ValidatorUtils;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.SDDManager;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.SDD_DAS;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.ValidationRule;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.XMLValidationError;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.exception.XMLValidationException;
import org.w3c.dom.Document;

public class SDDManagerImpl implements SDDManager {
	// List of Deployment and Package descriptors
	private Collection<Descriptor> descriptorsList = new Vector<Descriptor>();
	// List of ValidationRules
	private Collection<ValidationRule> ddValidationRules = new Vector<ValidationRule>();
	private Collection<ValidationRule> pdValidationRules = new Vector<ValidationRule>();
	// Encoding to use when writing out XML documents
	private String encoding = null;
	// Descriptor root element names 
	private static final String DD_ROOT_ELEMENT = "DeploymentDescriptor";
	private static final String PD_ROOT_ELEMENT = "PackageDescriptor"; 
	
	
	public void addDescriptorFromDocument(Document descriptorDocument, File descriptorFile)
			throws IllegalArgumentException {
		int descriptorType;
		SDD_DAS sddDas = null;
		
		// Check for null arguments
		if (null == descriptorDocument || null == descriptorFile) {
			throw new IllegalArgumentException();
		}
		
		// Determine the type of the descriptor, Deployment or Package
		if (ValidatorUtils.nodeNameEquals(descriptorDocument.getDocumentElement(), DD_ROOT_ELEMENT)) {
			descriptorType = DEPLOYMENT_DESCRIPTOR;
			
			// Create a DAS instance for this Descriptor and add existing ValidationRules to it
			sddDas = new SDD_DASImpl();
			for (ValidationRule rule : ddValidationRules) {
				sddDas.addValidation(rule);
			}
		}
		else if (ValidatorUtils.nodeNameEquals(descriptorDocument.getDocumentElement(), PD_ROOT_ELEMENT)) {
			descriptorType = PACKAGE_DESCRIPTOR;
			
			// Create a DAS instance for this Descriptor and add existing ValidationRules to it
			sddDas = new SDD_DASImpl();
			for (ValidationRule rule : pdValidationRules) {
				sddDas.addValidation(rule);
			}
		}
		else {
			// Descriptor isn't either of the expected types, throw an exception
			throw new IllegalArgumentException();
		}		
		
		// Add the Descriptor to the list
		descriptorsList.add(new Descriptor(descriptorDocument, descriptorFile, sddDas, descriptorType));
	}
	
	public Document addDescriptorFromFile(File descriptorFile)
		throws IllegalArgumentException, FileNotFoundException, IOException, XMLValidationException {
		SDD_DAS sddDas = null;

		if (null == descriptorFile) {
			throw new IllegalArgumentException();
		}
		
		// Create an SDD_DASImpl to load the Document for the given XML file
		SDD_DAS mySddDas = new SDD_DASImpl();
		Document document = mySddDas.loadDocument(new FileInputStream(descriptorFile));
		
		// Determine the type of the descriptor, Deployment or Package
		int descriptorType;
		if (ValidatorUtils.nodeNameEquals(document.getDocumentElement(), DD_ROOT_ELEMENT)) {
			descriptorType = DEPLOYMENT_DESCRIPTOR;
			
			// Create a DAS instance for this Descriptor and add existing ValidationRules to it
			sddDas = new SDD_DASImpl();
			for (ValidationRule rule : ddValidationRules) {
				sddDas.addValidation(rule);
			}
		}
		else if (ValidatorUtils.nodeNameEquals(document.getDocumentElement(), PD_ROOT_ELEMENT)) {
			descriptorType = PACKAGE_DESCRIPTOR;
			
			// Create a DAS instance for this Descriptor and add existing ValidationRules to it
			sddDas = new SDD_DASImpl();
			for (ValidationRule rule : pdValidationRules) {
				sddDas.addValidation(rule);
			}
		}
		else {
			// Descriptor isn't either of the expected types, throw an exception
			throw new IllegalArgumentException();
		}		
		
		// Add the newly loaded DataGraph
		descriptorsList.add(new Descriptor(document, descriptorFile, sddDas, descriptorType));
		
		return (document);
	}

	public void removeDescriptor(Document descriptorDocument) {
		// Find the Descriptor matching the given Document and remove it from the list
		for (Descriptor descriptor : descriptorsList) {
			if (descriptor.getDocument() == descriptorDocument) {
				descriptorsList.remove(descriptorDocument);
			}
		}
	}
	
	public void addValidation(ValidationRule validationRule, int descriptorType) throws IllegalArgumentException {
		if (null == validationRule) {
			throw new IllegalArgumentException();
		}
		if (descriptorType != DEPLOYMENT_DESCRIPTOR && descriptorType != PACKAGE_DESCRIPTOR) {
			throw new IllegalArgumentException();
		}
		
		// Add this validation rule to all descriptors of the given type
		for (Descriptor descriptor : descriptorsList) {
			if (descriptor.getType() == descriptorType) {
				descriptor.getDAS().addValidation(validationRule);
			}
			
		}
		
		// Add the rule to the appropriate list
		switch (descriptorType) {
		case DEPLOYMENT_DESCRIPTOR:
			ddValidationRules.add(validationRule);
			break;
		case PACKAGE_DESCRIPTOR:
			pdValidationRules.add(validationRule);
			break;
		}
	}

	public void removeValidation(ValidationRule validationRule, int descriptorType) throws IllegalArgumentException {
		if (descriptorType != DEPLOYMENT_DESCRIPTOR && descriptorType != PACKAGE_DESCRIPTOR) {
			throw new IllegalArgumentException();
		}
		
		// Remove this validation rule from all descriptors of the given type
		for (Descriptor descriptor : descriptorsList) {
			if (descriptor.getType() == descriptorType) {
				descriptor.getDAS().removeValidation(validationRule);
			}			
		}

		// Remove the rule from the appropriate list
		switch (descriptorType) {
		case DEPLOYMENT_DESCRIPTOR:
			ddValidationRules.remove(validationRule);
			break;
		case PACKAGE_DESCRIPTOR:
			pdValidationRules.remove(validationRule);
			break;
		}
	}

	public Collection<Document> getDeploymentDescriptors() {
		// Build a list of Deployment Descriptor Documents
		Collection<Document> documents = new Vector<Document>();
		
		for (Descriptor descriptor : descriptorsList) {
			// Only include Deployment Descriptors
			if (descriptor.getType() == DEPLOYMENT_DESCRIPTOR) {
				documents.add(descriptor.getDocument());	
			}			
		}
				
		return documents;
	}

	public Collection<Document> getPackageDescriptors() {
		// Build a list of Package Descriptor Documents
		Collection<Document> documents = new Vector<Document>();
		
		for (Descriptor descriptor : descriptorsList) {
			// Only include Deployment Descriptors
			if (descriptor.getType() == PACKAGE_DESCRIPTOR) {
				documents.add(descriptor.getDocument());	
			}			
		}
				
		return documents;
	}
	
	public Collection<Document> getAllDescriptors() {
		// Build a list of all Documents
		Collection<Document> documents = new Vector<Document>();
		
		for (Descriptor descriptor : descriptorsList) {
			documents.add(descriptor.getDocument());
		}
				
		return documents;		
	}

	public Collection<XMLValidationError> validateAllDescriptors() {
		// Call validateDescriptors() with the complete list of descriptors
		return validateDescriptors(getAllDescriptors());
	}

	public Collection<XMLValidationError> validateDescriptors(Collection<Document> descriptorDocs) {
		// Keep track of errors that occur during validation
		Collection<XMLValidationError> xmlValidationErrors = new Vector<XMLValidationError>();
		
		// Go through the list of Descriptors and validate any that are in the given list to validate
		for (Descriptor descriptor : descriptorsList) {
			for (Document descriptorDoc : descriptorDocs) {
				// Validate this descriptor if it's in the list to validate
				if (descriptor.getDocument() == descriptorDoc) {
					try {
						
						Collection<XMLValidationError> validationErrors =
							descriptor.getDAS().validate(descriptor.getDocument());
						
						// If there are any validation errors, add them to the list
						xmlValidationErrors.addAll(validationErrors);

					} catch (IllegalArgumentException e) {
						// All descriptors are checked when added to the SDDManager, so this shouldn't happen
					}
				}
			}
		}
		
		// Return the list of validation errors (empty if no errors occurred)
		return xmlValidationErrors;
	}

	public void writeAllDescriptors(boolean performValidation) throws XMLValidationException, IOException {
		// Call writeDescriptors() with the complete list of descriptors
		writeDescriptors(getAllDescriptors(), performValidation);
	}

	public void writeDescriptors(Collection<Document> descriptorDocs, boolean performValidation)
			throws XMLValidationException, IOException {
		// Keep track of errors that occur during validation
		Collection<XMLValidationError> xmlValidationErrors = new Vector<XMLValidationError>();

		// Go through the list of Descriptors and write any that are in the given list to write
		for (Descriptor descriptor : descriptorsList) {
			for (Document descriptorDoc : descriptorDocs) {
				// Write this descriptor if it's in the list to write
				if (descriptor.getDocument() == descriptorDoc) {
					try {
						
						descriptor.getDAS().saveDocument(descriptor.getDocument(),
								new FileOutputStream(descriptor.getXmlFile()), encoding, performValidation);
						
					} catch (XMLValidationException e) {
						
						// Add the errors from this validation to the list
						Iterator<XMLValidationError> errorIter = e.getValidationErrorIterator();
						
						while (errorIter.hasNext()) {
							xmlValidationErrors.add(errorIter.next());
						}						
					}
				}
			}
		}
		
		// If any validation errors occurred, throw an exception containing the list of errors
		if (!xmlValidationErrors.isEmpty()) {
			throw new XMLValidationException(xmlValidationErrors);
		}
	}
	
	public Collection<ValidationRule> getPackageDescriptorRules() {
		return pdValidationRules;
	}
	
	public Collection<ValidationRule> getDeploymentDescriptorRules() {
		return ddValidationRules;
	}

	public Collection<ValidationRule> getAllValidationRules() {
		Collection<ValidationRule> validationRules = new Vector<ValidationRule>();
		
		// Combine the Package and Deployment ValidationRule lists
		validationRules.addAll(ddValidationRules);
		validationRules.addAll(pdValidationRules);
		
		return validationRules;
	}
	
	public void setEncoding(String encoding) {
		this.encoding = encoding;
	}

	
	private class Descriptor {
		Document document;
		File xmlFile;
		SDD_DAS DAS;
		int type;
		
		public Descriptor(Document document, File xmlFile, SDD_DAS DAS, int type) {
			this.document = document;
			this.xmlFile = xmlFile;
			this.DAS = DAS;
			this.type = type;
		}

		public Document getDocument() {
			return document;
		}

		public void setDocument(Document document) {
			this.document = document;
		}

		public int getType() {
			return type;
		}

		public void setType(int type) {
			this.type = type;
		}

		public File getXmlFile() {
			return xmlFile;
		}

		public void setXmlFile(File xmlFile) {
			this.xmlFile = xmlFile;
		}

		public SDD_DAS getDAS() {
			return DAS;
		}

		public void setDAS(SDD_DAS DAS) {
			this.DAS = DAS;
		}
	}
}