/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.operations;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.cosmos.rm.internal.repository.RepositoryMessages;
import org.eclipse.cosmos.rm.internal.repository.SMLRepositoryUtil;
import org.eclipse.cosmos.rm.internal.repository.core.FileSystemSMLRepository;
import org.eclipse.cosmos.rm.internal.repository.core.IFileSystemSMLProperties;
import org.eclipse.cosmos.rm.internal.repository.core.MetadataProcessor;
import org.eclipse.cosmos.rm.internal.repository.resource.SMLFileMetadata;
import org.eclipse.cosmos.rm.internal.validation.common.SMLIFIdentity;
import org.eclipse.cosmos.rm.internal.validation.core.IValidator;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLValidateOperation;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDefinitionDocument;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLInstanceDocument;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata;
import org.eclipse.osgi.util.NLS;

/**
 * The implementation of the validate operation for this file-system based
 * repository.  This operation expects the following arguments:
 * <ol>
 * 	<li>A {@link Map} that stores the validation attributes (required) </li>
 * </ol>
 * The client is assumed to include all attributes that are required for a 
 * validation operation to run successfully.  At the minimum, the following 
 * attributes are needed:
 * <ul>
 * 	<li>{@link IValidator#ATTRIBUTE_ENV}</li>
 * 	<li>{@link IValidator#ATTRIBUTE_INSTANCE}</li>	
 * 	<li>{@link IValidator#ATTRIBUTE_INPUT_TYPE}</li>
 * </ul>
 * 
 * @author Ali Mehregani
 * @author John Arwe
 */
public class FileValidateOperation extends AbstractSMLOperation implements ISMLValidateOperation
{
	/**
	 * The validator - our entry point to the validation plug-in
	 */
	private SMLMainValidator validator;
	
	/**
	 * SML Units that will be validated together so as
	 * to hopefully be used together for relationship data.
	 */
	private Collection<ISMLDocument> smlUnitsToValidate;

	/**
	 * The status of this operation
	 */
	private boolean status;

	/**
	 * The internal errors while performing this operation
	 */
	private List<String> internalErrors;
	
	
	/**
	 * The constructor
	 * 
	 * @param repository The repository
	 */
	public FileValidateOperation(ISMLRepository repository)
	{
		super(repository);	
		validator = new SMLMainValidator();
		internalErrors = new ArrayList<String>();
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.operations.ISMLOperation#getId()
	 */
	public String getId()
	{
		return ISMLValidateOperation.ID;
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.operations.ISMLOperation#run()
	 */
	@SuppressWarnings("unchecked")
	public void run() throws RepositoryOperationException
	{
		final String[] REQUIRED_ATTRIBUTES = new String[]{
			IValidator.ATTRIBUTE_ENV,
			IValidator.ATTRIBUTE_INSTANCE,	
			IValidator.ATTRIBUTE_INPUT_TYPE
		};
		// Check the arguments
		Object[] arguments = getArguments();
		if ((arguments == null) || (arguments.length != 1 && super.checkType(arguments[0].getClass(), Map.class)))
		{
			throw new RepositoryOperationException(RepositoryMessages.operationBadArguments);
		}
		Map<String, Object> argumentAttributes = (Map<String, Object>)arguments[0];
		for (int i = 0; i < REQUIRED_ATTRIBUTES.length; i++)
		{
			if (argumentAttributes.get(REQUIRED_ATTRIBUTES[i]) == null)
				throw new RepositoryOperationException(RepositoryMessages.operationBadArguments);
		}
		
		smlUnitsToValidate = new ArrayList<ISMLDocument>();
		Map<String, Object> attributes = validator.getAttributes();

		// Add the context directory to the validator attribute
		attributes.put(IValidator.ATTRIBUTE_REPOSITORY_ROOT, getRepository().getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, ""));
		
		// Copy the argument into the validator attributes
		for (Iterator<String> attKeys = argumentAttributes.keySet().iterator(); attKeys.hasNext();)
		{
			String key = attKeys.next();
			attributes.put(key, argumentAttributes.get(key));
		}		
				
		StringTokenizer instances = new StringTokenizer((String)attributes.get(IValidator.ATTRIBUTE_INSTANCE), ",");		
		boolean smlifInputType = IValidator.VALUE_SML_IF.equals(attributes.get(IValidator.ATTRIBUTE_INPUT_TYPE));
		SMLFileMetadata metadata = new SMLFileMetadata();		
		
		status = true;
		for (int i = 0; instances.hasMoreTokens(); i++) 
		{
			// If it's just SML-IF files that we need to validate, then just call validate
			if (smlifInputType)
			{
				attributes.put(IValidator.ATTRIBUTE_INSTANCE, instances.nextToken());
				validator.initialize(attributes);
				status = validator.start() && status;
			}
			// Otherwise it's SML model units we're validating
			else
			{
				metadata.setId(instances.nextToken());
				ISMLDocument[] document = getRepository().fetchDocuments(metadata);
				
				// Only one document is expected to be fetched
				if (document.length != 1)
				{
					addInternalError(NLS.bind(RepositoryMessages.validateErrorMissingDocument, metadata.getId()));
				}
				
				// Add the document
				addSMLUnit(document[0]);
			}
		}	
		
		// Validate the SML model units
		if (!smlifInputType)
		{
			StringBuffer paths = new StringBuffer();			
			for (Iterator<ISMLDocument> documents = smlUnitsToValidate.iterator(); documents.hasNext();)
			{
				ISMLDocument document = documents.next();
				paths.append(SMLRepositoryUtil.getDocumentPath(document)).append(",");
			}
			
			if (paths.length() > 0)
			{
				// Remove the last column
				paths.deleteCharAt(paths.length() - 1);
				attributes.put(IValidator.ATTRIBUTE_INSTANCE, paths.toString());
				
				
				if (getRepository() instanceof FileSystemSMLRepository)
				{					
					MetadataProcessor metadataProcessor = ((FileSystemSMLRepository)getRepository()).getMetadataProcessor();
					
					// Retrieve the identity information and set it as an attribute
					attributes.put(IValidator.ATTRIBUTE_IDENTITY, metadataProcessor.getMetadataHandler().getIdentity());
					//	Temporary fix to allow validation tests for SML Units to run
					String repoRoot = getRepository().getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, "");
					SMLIFIdentity modelIdentity = metadataProcessor.getMetadataHandler().getIdentity();
					String modelBaseURI = modelIdentity.getBaseURI();
					if (modelBaseURI == null)
					{
						try
						{	//	URI constructor will handle quoting of all path characters we might want except for @, so that is done manually
							modelBaseURI = (new URI("file","localhost","/"+repoRoot.replaceAll("@", "%40"),null,null)).toString();	
						}
						catch (URISyntaxException e)
						{
							modelBaseURI = "";	//	TODO Surface the problem? Best we can do for now is revert to old behavior
						}
						modelIdentity.setBaseURI(modelBaseURI);
						attributes.put(IValidator.ATTRIBUTE_IDENTITY, modelIdentity);
					}
										
					// Retrieve the rule binding information and set it as an attribute
					attributes.put(IValidator.ATTRIBUTE_RULE_BINDINGS, metadataProcessor.getMetadataHandler().getAliasRuleBinding());
					
					// Retrieve the alias information and set it as an attribute
					attributes.put(IValidator.ATTRIBUTE_ALIASES, metadataProcessor.getMetadataHandler().getFileAliasMap());
				}
 
				
				validator.initialize(attributes);
				status = validator.start() && status;				
			}			
		}
	}	
	
	private void addInternalError(String errorMessage)
	{
		internalErrors.add(errorMessage);
	}
	
	
	private void addInternalError(RepositoryOperationException e)
	{		
		internalErrors.add(e.getLocalizedMessage());
		StringWriter sw = new StringWriter();
		e.printStackTrace(new PrintWriter(sw));			
		internalErrors.add(sw.getBuffer().toString());
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.operations.ISMLValidateOperation#getInternalErrors()
	 */
	public String[] getInternalErrors()
	{
		return (String[])internalErrors.toArray(new String[internalErrors.size()]);
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.operations.ISMLValidateOperation#getStatus()
	 */
	public boolean getStatus()
	{
		return status;
	}
	
	private void addSMLUnit(ISMLDocument document)
	{		
		if ((document == null) || smlUnitsToValidate.contains(document))
			return;
	
		smlUnitsToValidate.add(document);
		if (ISMLMetadata.DOCUMENT_TYPE_INSTANCE == document.getMetadata().getDocumentType())
		{
			ISMLInstanceDocument instance = (ISMLInstanceDocument)document;
			try
			{
				ISMLDefinitionDocument[] definitions = instance.retrieveDefinitions();
				for (int i = 0; i < definitions.length; i++) {
					addSMLUnit(definitions[i]);
				}
				ISMLDocument[] references = instance.retrieveReferences(new SMLFileMetadata());
				for (int i = 0; i < references.length; i++)
				{
					addSMLUnit(references[i]);
				}
			} 
			catch (RepositoryOperationException e)
			{
				addInternalError(e);
			}			
		}
	}
}
