/***********************************************************************
 * Copyright (c) 2009 SAS Institute, Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SAS Institute - Initial API and implementation
 ***********************************************************************/
package org.eclipse.cosmos.me.sdd.profile.cim.handler;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.PrintStream;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.eclipse.cosmos.me.sdd.cr.IResourceHandler;
import org.eclipse.cosmos.me.sdd.profile.cim.J2eeJVMType;
import org.eclipse.cosmos.me.sdd.profile.cim.Version;
import org.eclipse.cosmos.me.sdd.schema.CapacityConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConditionalPropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConsumptionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.PropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.RequirementResourceConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ResourceType;
import org.eclipse.cosmos.me.sdd.schema.VersionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.VersionConstraintValueType;
import org.eclipse.cosmos.me.sdd.schema.VersionRangeType;
import org.eclipse.cosmos.me.sdd.schema.VersionValueType;
import org.eclipse.cosmos.me.sdd.schema.ext.SDDContext;

import com.sas.tools.deployjni.JNIUtils;
import com.sas.tools.deployjni.Utilities;
import com.sas.tools.deployjni.winregistry.WinRegAPI;

public class JREHandler extends J2eeJVMType implements IResourceHandler {
	
	static final String SUN = "sun";
	static final String IBM = "IBM";
	static final String BEA = "BEA";

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			CapacityConstraintType constraint) {
		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			ConsumptionConstraintType constraint) {
		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			PropertyConstraintType constraint) {
		
		if (constraint.getPropertyName().equals(VENDOR)) {
			VENDOR_VALUE = constraint.getValue();
			return resolveVendor(VENDOR_VALUE);
		}
		else return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			ConditionalPropertyConstraintType constraint) {
		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			VersionConstraintType constraint) {
		
		List <String> versions = (List)getVersionValue(ctxt, rsrc);
		Iterator <String> versionIter = versions.iterator();
		while (versionIter.hasNext()) {
			
			Version version = new Version((String)versionIter.next());
			VersionConstraintValueType supported = constraint.getSupported();
			if (supported != null) {
				// match individual values
				for (VersionValueType value : supported.getValue()) {
					Version v = new Version(value.getVersion());
					if (version.compareTo(v) == 0) {
						return true;
					}
				}
				
				// match ranges
				for (VersionRangeType range : supported.getRange()) {
					if (range.getMinVersion() != null) {
						Version v = new Version(range.getMinVersion());
						if (version.compareTo(v) < 0) {
							continue; // version too low
						}
					}
					
					if (range.getMaxVersion() != null) {
						Version v = new Version(range.getMaxVersion().getValue());
						if (version.compareTo(v) > 0) {
							continue; // version too high
						}
					}
					return true; // passed both min and max tests
				}
			}
		}
		return false;
	}

	public Object getPropertyValue(SDDContext ctxt, ResourceType rsrc,
			QName qname) {
		return null;
	}

	public Object getVersionValue(SDDContext ctxt, ResourceType rsrc) {
		
		List <String> versions = new ArrayList<String>();
	
		if (Utilities.getOS() == Utilities.OS_WINDOWS) {
			if (JNIUtils.libraryLoaded(getCurrentPath(), WinRegAPI.libNames)) {
				
				String [] sunVersions = WinRegAPI.childrenForKey32(WinRegAPI.HKEY_LOCAL_MACHINE,"SOFTWARE\\JavaSoft\\Java Runtime Environment\\");
				for (int i=0;i<sunVersions.length;i++)
				{
					versions.add(sunVersions[i]);
				}
				String [] BEAVersions =  WinRegAPI.childrenForKey32(WinRegAPI.HKEY_LOCAL_MACHINE,"SOFTWARE\\JRockit\\Java Runtime Environment\\");
				for (int i=0;i<BEAVersions.length;i++)
				{
					versions.add(BEAVersions[i]);
				}
			}
		}
		else if (Utilities.getOS() == Utilities.OS_LINUX) {
			
			try {
                // Create a shell script to run java -fullversion and capture
                // the output to a file that can be parsed for the version.
                String      output     = ".jrever";
                String      script     = "jrescript";
                File        scriptFile = new File(script);
                PrintStream out        = new PrintStream(new FileOutputStream(scriptFile));

                out.println("#!/bin/sh");
                out.println("if [ -f " + output + " ]; then ");
                out.println("  rm -f " + output);
                out.println("fi");
                out.println("export LANG=en_US");
                out.println("java -version 1>" + output + " 2>&1");
                out.close();

                // Make the script executable
                Process scriptProc = Runtime.getRuntime().exec("chmod 755 " + script);
                scriptProc.waitFor();
                scriptProc = null;

                String  command = "/bin/sh -c ./" + script;
                Process javaVer = Runtime.getRuntime().exec(command);
                javaVer.waitFor();

                scriptFile.delete();

                // Look in output for the version string
                File outputFile = new File(output);
                if (outputFile.isFile()) {
                   BufferedReader in = new BufferedReader(new FileReader(outputFile));

                   String line = in.readLine();
                   if (line != null) {
                      int firstQuote = line.indexOf("\"");
                      int lastQuote  = line.indexOf("\"", firstQuote + 1);

                      if ((firstQuote > -1) && (lastQuote > -1))
                        versions.add(line.substring(firstQuote + 1, lastQuote));
                   }
                   in.close();
                   outputFile.delete();
                }
             }
             catch (Exception e) {
            	 e.printStackTrace();
             }
			
		}
		return versions;
	}
	
	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			RequirementResourceConstraintType constraint) {
		boolean result = true;
		for (Object o : constraint.getResourceConstraintGroup()) {
			if (o instanceof JAXBElement) {
				Object value = ((JAXBElement<?>)o).getValue();
				if (value instanceof PropertyConstraintType) {
					result = result && evaluate(ctxt, rsrc, (PropertyConstraintType)value);
					continue;
				}
				if (value instanceof VersionConstraintType) {
					result = result && evaluate(ctxt, rsrc, (VersionConstraintType)value);
				}
			}
		}
		
		return result;
	}
	
	private static boolean resolveVendor(String vendor) {

		if (Utilities.getOS() == Utilities.OS_WINDOWS) {
			if (JNIUtils.libraryLoaded(getCurrentPath(), WinRegAPI.libNames)) {
				if (vendor.equalsIgnoreCase(SUN)) {
					if (WinRegAPI.isWindowsRegistryKey32(
							WinRegAPI.HKEY_LOCAL_MACHINE,
							"SOFTWARE\\JavaSoft\\Java Runtime Environment\\")) {
						return true;
					}
				}
				else if (vendor.equalsIgnoreCase(IBM)) {
					if (WinRegAPI
							.getWindowsRegistryValue(
									WinRegAPI.HKEY_LOCAL_MACHINE,
									"SOFTWARE\\IBM\\MQSeries\\CurrentVersion\\Components",
									"Server") != null) {
						return true;
					}
				}
				else if (vendor.equalsIgnoreCase(BEA)) {
					if (WinRegAPI.isWindowsRegistryKey32(
							WinRegAPI.HKEY_LOCAL_MACHINE,
							"SOFTWARE\\JRockit\\Java Runtime Environment\\")) {
						return true;
					}
				}
			}
		}
		else if (Utilities.getOS() == Utilities.OS_LINUX) {
			if (vendor.equalsIgnoreCase(SUN)) {
				for (int i = 0; i < unixPatterns.length; i++) {
					String s = unixFind(MessageFormat.format(unixPatterns[i],
							"JRE", "", "SUN"));
					if (s != null) {
						return true;
					}
				}
			}
			else if (vendor.equalsIgnoreCase(IBM)) {
				for (int i = 0; i < unixPatterns.length; i++) {
					String s = unixFind(MessageFormat.format(unixPatterns[i],
							"JRE", "", "IBM"));
					if (s != null) {
						return true;
					}
				}
			}
			else if (vendor.equalsIgnoreCase(BEA)) {
				for (int i = 0; i < unixPatterns.length; i++) {
					String s = unixFind(MessageFormat.format(unixPatterns[i],
							"JRE", "", "BEA"));
					if (s != null) {
						return true;
					}
				}
			}
		}

		return false;
	}

}
