/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.reference;

import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.NamespaceContext;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * An abstract implementation of the {@link IXScheme} interface
 * 
 * @author Ali Mehregani
 */
public abstract class AbstractScheme implements IXScheme
{
	/**
	 * The xpath object used in compiling and evaluating XPath expressions
	 */
	private static XPath xpath;	
	
	/**
	 * The expression of this scheme
	 */
	private String expression;
	
	/**
	 * The namespace context to be used when evaluating the
	 * XPath expression
	 */
	private NamespaceContext namespaceContext;
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.reference.IXScheme#getExpression()
	 */
	public String getExpression()
	{
		return expression;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.reference.IXScheme#setExpression(java.lang.String)
	 */
	public void setExpression(String expression)
	{
		this.expression = expression;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.reference.IXScheme#evaluate(java.lang.Object)
	 */
	public Object evaluate(Object context) throws BadExpressionException
	{
		if (xpath == null)
		{
			XPathFactory factory = XPathFactory.newInstance();
			if (factory == null)
				throw new BadExpressionException(SMLValidationMessages.errorMissingXPathFactory);
			
		    xpath = factory.newXPath();		
		    if (xpath == null)
		    	throw new BadExpressionException(SMLValidationMessages.errorMissingXPathInstance);
		}
              
        if (getNamespaceContext() != null)
        	xpath.setNamespaceContext(getNamespaceContext());
        
        List<NodeList> listOfNodeList = new ArrayList<NodeList>();
        try
        {
        	String xpathExpStr = getExpression();
        	if (xpathExpStr != null)
        		xpathExpStr = xpathExpStr.replace('', '\'');
	        XPathExpression xpathExpr = xpath.compile(xpathExpStr);
	        if (xpathExpr == null)
	        	throw new BadExpressionException (SMLValidationMessages.errorBadXPathExpression);
	        	        
	        if (context instanceof Node)
	        {
	        	evaluateXPathExpression(xpathExpr, (Node)context, listOfNodeList);       	
	        }
	        else if (context instanceof NodeList[])
	        {
	        	NodeList[] contextNodeList = (NodeList[])context;
	        	
	        	/* For every node list */
	        	for (int i = 0; i < contextNodeList.length; i++)
				{
					/* For every node in the node list */
	        		for (int j = 0, nodeListCount = contextNodeList[i].getLength(); j < nodeListCount; j++)
					{
	        			evaluateXPathExpression(xpathExpr, contextNodeList[i].item(j), listOfNodeList);       	
					}
				}
	        }
        }
        catch (XPathExpressionException xe)
        {
        	throw new BadExpressionException(xe);        	
        }
          
        return listOfNodeList.toArray(new NodeList[listOfNodeList.size()]);
	}
	
	
	private void evaluateXPathExpression(XPathExpression xpathExpr, Node context, List<NodeList> listOfNodeList) throws XPathExpressionException
	{
    	NodeList nodeList = (NodeList)xpathExpr.evaluate(context, XPathConstants.NODESET);
    	if (nodeList != null)
    		listOfNodeList.add(nodeList); 
	}


	/**
	 * @return the namespaceContext
	 */
	public NamespaceContext getNamespaceContext()
	{
		return namespaceContext;
	}


	/**
	 * @param namespaceContext the namespaceContext to set
	 */
	public void setNamespaceContext(NamespaceContext namespaceContext)
	{
		this.namespaceContext = namespaceContext;
	}
}
