/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.provisional.repository.operations;

import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;

/**
 * Represents an operation that can be performed on a repository.
 * {@link ISMLRepository#getSupportedOperations()} should be used to retrieve
 * the set of operations that a repository will support. 
 * 
 * 
 * @author Ali Mehregani
 */
public interface ISMLOperation
{
	/**
	 * Returns the unique ID of this operation.  The ID is expected
	 * to be unique amongst the supported operations of a repository
	 * 
	 * @return The unique ID of this operation
	 */
    public String getId();
    
    
    /**
     * Sets the arguments of this operation.  The type of the argument
     * is a contract between the implementation of the operation and 
     * the consuming client.
     *  
     * @param arguments The arguments of this operation
     * @throws IllegalArgumentException If the argument type is not what is 
     * expected by the underlying implementation
     */
    public void setArguments(Object[] arguments) throws IllegalArgumentException;
    
    
    /**
     * Adds an argument to this operation.  The type of the argument
     * is a contract between the implementation of the operation and 
     * the consuming client.
     * 
     * @param argument The argument to be added
     * @throws IllegalArgumentException If the argument type is not what is 
     * expected by the underlying implementation
     */
    public void addArgument(Object argument) throws IllegalArgumentException;
    
    
    /**
     * Adds listener to the list of operation progress listeners that are
     * tracked by this operation
     * 
     * @param listener The listener to be added
     */
    public void addProgressListener(IProgressListener listener);
    
    
    /**
     * Removes listener from the list of progress listener.  This method
     * has to effect if listener is not registered with this operation
     * 
     * @param listener The listener to be removed
     */
    public void removeProgressListener(IProgressListener listener);
    
        
    /**
     * Executes this operation.
     * @throws RepositoryOperationException In case of any error
     */
    public void run() throws RepositoryOperationException;
    
    
    /**
     * Used to retrieve the output of this operation if one is expected to
     * be produced.  Not all operations are obligated to produce an output.
     * The type of the output should be specified as part of the operation's
     * contract.  An operation will need to be executed using this{@link #run()}
     * before its output is retrieved.
     *  
     * @return The output of this operation
     */
    public Object getOutput();
        
}
