/*******************************************************************************
 * Copyright (c) 2009 SAS Institute, Inc., and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SAS Institute, Inc. - initial API and implementation
 *     SAS Institute, Inc. - Bug 275699
 *     SAS Institute, Inc. - Bug 275673
 *     SAS Institute, Inc. - Bug 
 ******************************************************************************/
package org.eclipse.cosmos.me.sdd.cli;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URI;
import java.util.List;
import java.text.MessageFormat;
import java.util.Properties;

import org.eclipse.cosmos.me.sdd.advisor.IAdvisor;
import org.eclipse.cosmos.me.sdd.schema.BooleanParameterType;
import org.eclipse.cosmos.me.sdd.schema.BoundaryType;
import org.eclipse.cosmos.me.sdd.schema.IntegerParameterType;
import org.eclipse.cosmos.me.sdd.schema.RequirementResourceConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ResourceType;
import org.eclipse.cosmos.me.sdd.schema.StringParameterType;
import org.eclipse.cosmos.me.sdd.schema.URIParameterType;

public class CLIAdvisor implements IAdvisor {

	private static final String ONE = "1";
	private static final String ZERO = "0";
	private static final String TRUE = "true";
	private static final String FALSE = "false";
	
	private BufferedReader br;
	private Properties properties;

	public CLIAdvisor() {
		br = new BufferedReader(new InputStreamReader(System.in));
	}
	
	public void setProperties(Properties descriptorLanguageBundle) {
		properties = descriptorLanguageBundle;
	}

	public void displayMessage(String message) {
		System.out.println(message);
	}

	public Integer requestParameter(IntegerParameterType ipt) {
		
		String name = properties != null ? ipt.getDescription().getTranslationKey() : null; 
		if (name != null) {
			name = getPropertyValue(name);
		}
		
		String value = readVariable(name != null ? name : null, ipt.getDefaultValue());
		
		Integer returnVal = null;
		// try to parse Integer from string, if unable to do so, repeat
		try {
			returnVal = Integer.parseInt(value);
			
			// See if this type has defined any bounds that have to be checked.
			List<BoundaryType> bounds = ipt.getBounds();
			if(bounds != null) {
				for(BoundaryType bound : bounds) {
					String upper = bound.getUpperBound();
					String lower = bound.getLowerBound();
					
					if(upper != null) {
						if(returnVal >(Integer.parseInt(upper))) {
							displayMessage(Messages.getString("CLIAdvisor.UpperBoundExceeded", name)); 
							return (requestParameter(ipt));
						}
					}
					
					if(lower != null) {
						if(returnVal <= (Integer.parseInt(lower))) {
							displayMessage(Messages.getString("CLIAdvisor.LowerBoundExceeded", name)); 
							return (requestParameter(ipt));
						}
					}
				}
			}
		} catch (Exception e) {
			displayMessage(Messages.getString("CLIAdvisor.ProvideEntry")); 
			return (requestParameter(ipt));
		}
		return returnVal;
	}

	public String requestParameter(StringParameterType spt) {

		String name = properties != null ? spt.getDescription().getTranslationKey() : null; 
		if (name != null) {
			name = getPropertyValue(name);
		}
		
		return readVariable(name != null ? name : null, spt.getDefaultValue());
	}

	public Boolean requestParameter(BooleanParameterType bpt) {
		String name = properties != null ? bpt.getDescription().getTranslationKey() : null; 
		if (name != null){
			name = getPropertyValue(name);
		}
		
		String value = readVariable(name != null ? name : null, bpt.getDefaultValue());

		// try to parse boolean from string, if unable to do so, repeat
		if (value.equalsIgnoreCase(CLIAdvisor.ONE))
			return true;
		else if (value.equalsIgnoreCase(CLIAdvisor.ZERO))
			return false;
		else if (value.equalsIgnoreCase(CLIAdvisor.TRUE))
			return true;
		else if (value.equalsIgnoreCase(CLIAdvisor.FALSE))
			return false;
		else {
			displayMessage(Messages.getString("CLIAdvisor.ProvideEntry")); 
			return (requestParameter(bpt));
		}
	}
	
	

	public URI requestParameter(URIParameterType upt) {
		
		String name = properties != null ? upt.getDescription().getTranslationKey() : null; 
		if (name != null)
		{
			name = getPropertyValue(name);
		}
		
		URI returnVal = null;
		// try to parse URI from string, if unable to do so, repeat
		try {
			returnVal = URI.create(readVariable(name != null ? name : null, upt.getDefaultValue()));
		} catch (Exception e) {
			return requestParameter(upt);
		}
		return returnVal;
	}

	// method simply takes in a name and a default value
	// and returns the users input
	private String readVariable(String name, String defaultVal) {
		
		boolean hasDefault = false;

		if(name == null)
			name = Messages.getString("CLIAdvisor.MissingDisplayName");
			
		// if there is a defaultValue, display it and give the option to use it
		if ((defaultVal != null) && (defaultVal.length() > 0)) {
			hasDefault = true;
			System.out.print(MessageFormat.format(name + "({0})",defaultVal));
		}
		else {
			System.out.print(name);
		}

		String value = null;
		try {
			value = br.readLine();
		} catch (IOException e) {
			e.printStackTrace();
		}

		// if they press enter and there is a default value, return it,
		// otherwise, ask again.
		if ((value == null) || (value.length() == 0)) {
			if (hasDefault)
				value = defaultVal;
			else
				value = readVariable(name, defaultVal);
		}

		return value;
	}

	public void displayError(RequirementResourceConstraintType resource) {
		if ((resource==null)||(resource.getResourceRef()==null))
			displayMessage(Messages.getString("CLIAdvisor.FailedResolution"));
		else {
			ResourceType rsrc = (ResourceType) resource.getResourceRef();
			displayMessage(Messages.getString("CLIAdvisor.FailedResolution", resource.getId() , rsrc.getName()));
		}
	}

	private String getPropertyValue(String translationKey)
	{	
		String [] allKeys = translationKey.split(",");
		
		if (allKeys.length==0)
			return null;
		if (allKeys.length==1)
			return properties.getProperty(allKeys[0]);
		else {
			String [] substitutions = new String [allKeys.length-1];
			String initialValue = properties.getProperty(allKeys[0]);
			for (int i=1; i<allKeys.length; i++) {
				substitutions[i-1]=properties.getProperty(allKeys[i]);
			}
			return MessageFormat.format(initialValue, (Object[])substitutions);
		}
	}
	
}
