/***********************************************************************
 * Copyright (c) 2009 CA, Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     CA, Inc. - Initial implementation
 *     SAS Institute, Inc. - Bug 275731
 ***********************************************************************/
package org.eclipse.cosmos.me.sdd.profile.cim.handler;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.text.MessageFormat;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.eclipse.cosmos.me.sdd.cr.IResourceHandler;
import org.eclipse.cosmos.me.sdd.schema.CapacityConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConditionalPropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConsumptionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConsumptionConstraintValueType;
import org.eclipse.cosmos.me.sdd.schema.PropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.RequirementResourceConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ResourceType;
import org.eclipse.cosmos.me.sdd.schema.VersionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ext.SDDContext;

public class FileSystemHandler implements IResourceHandler {
	static final QName FS = new QName("http://docs.oasis-open.org/sdd/ns/cim-profile", "CIM_FileSystem");
	static final QName FS_AvailableSpace = new QName("http://docs.oasis-open.org/sdd/ns/cim-profile", "CIM_FileSystem.AvailableSpace");
	static final String CMD_DIR = "cmd.exe /c dir \"{0}\"";
	static final String CMD_DF = "df -k {0}";
	static final String Windows = "Windows";
	static final Pattern PATTERN_DIR = Pattern.compile("([0-9,]+) bytes free");
	static final Pattern PATTERN_DF = Pattern.compile("(.*)\\s+([0-9]+)\\s+([0-9]+)\\s+([0-9]+)\\s+");
	static final String OS_NAME = "os.name";
	
	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			CapacityConstraintType constraint) {
		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			ConsumptionConstraintType constraint) {
		if (FS_AvailableSpace.equals(constraint.getPropertyName())) {
			ConsumptionConstraintValueType value = constraint.getValue();
			Long limit = Long.parseLong(value.getValue());
			if (value.getUnit() != null) {
				if (value.getUnit().equals("512-blocks")) {
					limit = limit * 512;
				}
			}
			
			Long freeSpace = (Long)getPropertyValue(ctxt, rsrc, constraint.getPropertyName());
			
			return (freeSpace.compareTo(limit) >= 0);
			
		}
		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			PropertyConstraintType constraint) {
		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			ConditionalPropertyConstraintType constraint) {
		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			VersionConstraintType constraint) {
		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			RequirementResourceConstraintType constraint) {
		boolean result = true;
		for (Object o : constraint.getResourceConstraintGroup()) {
			if (o instanceof JAXBElement) {
				Object value = ((JAXBElement<?>)o).getValue();
				if (value instanceof PropertyConstraintType) {
					result = result && evaluate(ctxt, rsrc, (PropertyConstraintType)value);
					continue;
				}
				if (value instanceof ConsumptionConstraintType) {
					result = result && evaluate(ctxt, rsrc, (ConsumptionConstraintType)value);
					continue;
				}
				if (value instanceof VersionConstraintType) {
					result = result && evaluate(ctxt, rsrc, (VersionConstraintType)value);
				}
			}
		}
		
		return result;
	}
	
	public Object getPropertyValue(SDDContext ctxt, ResourceType rsrc,
			QName qname) {
		if (FS.equals(rsrc.getType())) {
			if (FS_AvailableSpace.equals(qname)) {
				return getFreeSpace(ctxt.substituteVariables(rsrc.getName()));
			}
		}
		return null;
	}

	public Object getVersionValue(SDDContext ctxt, ResourceType rsrc) {
		return null;
	}

	/**
	 * Try and locate the amount of space on the requested machine.
	 */ 
	private Long getFreeSpace(String directoryName) { 
		try 
		{ 
			// Get the machine we are on.
			String os = System.getProperty(OS_NAME); 
			String command; 
			
			// Check the operating system.
			if (os.contains(Windows)) {
				command = MessageFormat.format(CMD_DIR, directoryName);
			} 
			else { 
				command = MessageFormat.format(CMD_DF, directoryName);
			} 
			
			// Run the command.
			Runtime runtime = Runtime.getRuntime(); 
			Process process = runtime.exec(command); 
			
			
			if (process == null) { 
				return Long.valueOf(0); 
			} 
			
			// Read the output of the command.
			BufferedReader inStream = new BufferedReader(new InputStreamReader(process.getInputStream())); 
			String line = null; 
			String output = null;
			while ((line = inStream.readLine()) != null) { 
				output = line; 
			} 
			
			// Determine which one to run...
			if (os.contains(Windows)) { 
				return windowsFreeSpace(output);
			} 
			else { 
				return unixFreeSpace(output);
			}  
		} 
		catch (Exception e) { 
			return Long.valueOf(0); 
		} 
	}

	private Long unixFreeSpace(String s) {
		// e.g. "C:\cygwin             97618972  50573164  47045808  52% /"
		
		// Make sure we got something.
		if (s != null) {
			Matcher m = PATTERN_DF.matcher(s);
			if (m.find()) {
				return (Long.parseLong(m.group(4)) * 1024); // assuming 1K blocks
			}
		}
		return Long.valueOf(0);
	}
	
	private Long windowsFreeSpace(String s) {
		// e.g. "             115 Dir(s)  48,175,329,280 bytes free"
		
		// Make sure we got something.
		if (s != null) {
			Matcher m = PATTERN_DIR.matcher(s);
			if (m.find()) {
				return Long.parseLong(m.group(1).replace(",", ""));
			}
		}
		return Long.valueOf(0);
	}


}
