/***********************************************************************
 * Copyright (c) 2009 CA, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     CA, Inc. - Initial implementation
 *     SAS Institute - Bug 275669 
 ***********************************************************************/
package org.eclipse.cosmos.me.sdd.o10r.impl;


import org.osgi.service.log.LogReaderService;
import org.osgi.service.log.LogService;


import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.List;
import java.util.Locale;
import java.util.Properties;


import org.eclipse.cosmos.me.sdd.advisor.IAdvisor;
import org.eclipse.cosmos.me.sdd.cr.ResourceHandler;
import org.eclipse.cosmos.me.sdd.log.LogHandler;
import org.eclipse.cosmos.me.sdd.o10r.Orchestrator;
import org.eclipse.cosmos.me.sdd.op.OperationHandler;
import org.eclipse.cosmos.me.sdd.schema.AdditionalContentType;
import org.eclipse.cosmos.me.sdd.schema.ArtifactType;
import org.eclipse.cosmos.me.sdd.schema.BooleanParameterType;
import org.eclipse.cosmos.me.sdd.schema.ContentType;
import org.eclipse.cosmos.me.sdd.schema.DeploymentDescriptorType;
import org.eclipse.cosmos.me.sdd.schema.InstallableUnitType;
import org.eclipse.cosmos.me.sdd.schema.InstallationArtifactsType;
import org.eclipse.cosmos.me.sdd.schema.IntegerParameterType;
import org.eclipse.cosmos.me.sdd.schema.OperationType;
import org.eclipse.cosmos.me.sdd.schema.PackageDescriptorType;
import org.eclipse.cosmos.me.sdd.schema.StringParameterType;
import org.eclipse.cosmos.me.sdd.schema.URIParameterType;
import org.eclipse.cosmos.me.sdd.schema.ext.SDDContext;
import org.eclipse.cosmos.me.sdd.schema.marshal.Marshaller;

public class OrchestratorImpl implements Orchestrator {
	
	private final String PURPOSE_DESCRIPTION_LANGUAGE_BUNDLE = "descriptorLanguageBundle";
	private final String DEFAULT_COUNTRY = "US";
	private final String DEFAULT_LANGUAGE = "en";
	
	private Marshaller marshaller;
	private OperationHandler ophandler;
	private ResourceHandler rsrcHandler;
	private IAdvisor advisor;
	private LogHandler log;
	private LogService logService;
	private LogReaderService logReader;
	
	public void setMarshaller(Marshaller marshaller) {
		this.marshaller = marshaller;
	}
	
	public void unsetMarshaller(Marshaller marshaller) {
		this.marshaller = null;
	}
	
	public void setOperationHandler(OperationHandler ophandler) {
		this.ophandler = ophandler;
	}
	
	public void unsetOperationHandler(OperationHandler ophandler) {
		this.ophandler = null;
	}
	
	public void setResourceHandler(ResourceHandler rsrcHandler) {
		this.rsrcHandler = rsrcHandler;
	}
	
	public void unsetResourceHandler(ResourceHandler rsrcHandler) {
		this.rsrcHandler = null;
	}
	
	public void setAdvisor(IAdvisor advisor) {
		this.advisor = advisor;
	}
	
	public void unsetAdvisor(IAdvisor advisor) {
		this.advisor = null;
	}
	
	public void setLogService(LogService logService) {
		this.logService = logService;
	}
	
	public void unsetLogService(LogService logService) {
		this.logService = null;
	}
	
	public void setLogReader(LogReaderService logReader) {
		this.logReader = logReader;
	}
	
	public void unsetLogReader(LogReaderService logReader) {
		this.logReader = null;
	}
	
	public void setLog(LogHandler log) {
		this.log = log;
	}
	
	public void unsetLog(LogHandler log) {
		this.log = null;
	}
	
	
	public void deploy(String packageUri, String operation) {
		OperationType op = OperationType.fromValue(operation);
		switch (op) {
		case INSTALL:
		case UNINSTALL:
			installableUnit(packageUri, op);
			break;
		default:
			break;
		}
	}

	private void installableUnit(String packageUri, OperationType op) {
		SDDContext ctxt = new SddContextImpl(marshaller, packageUri);
		PackageDescriptorType pd = ctxt.getPackageDescriptor();
		if (pd != null) {
			DeploymentDescriptorType dd = ctxt.getDeploymentDescriptor();
			if (dd != null) {
				
				if(installableUnitResolution(ctxt)) {
					InstallableUnitType unit = dd.getInstallableUnit();
					if (unit != null) {
						if (unit.getArtifacts() != null) {
							InstallationArtifactsType artifacts = unit.getArtifacts();
							switch (op) {
							case INSTALL:
								if (artifacts.getInstallArtifact() != null) {
									ArtifactType artifact = artifacts.getInstallArtifact();
									ophandler.invokeAction(ctxt, artifact, op);
									for (AdditionalContentType c : artifact.getAdditionalContent()) {
										ophandler.invokeAction(ctxt, artifact, c, op);
									}
								}
								break;
							case UNINSTALL:
								if (artifacts.getUninstallArtifact() != null) {
									ophandler.invokeAction(ctxt, artifacts.getUninstallArtifact(), op);
								}
								break;
							}
						}
					}
				}
			}
		}
	}
	
	private Properties getDescriptorLanguageBundle(SDDContext ctxt)
	{
		List <ContentType> contents = ctxt.getPackageDescriptor().getContents().getContent();
		for (ContentType c : contents)
		{
			if (c.getPurpose().equalsIgnoreCase(PURPOSE_DESCRIPTION_LANGUAGE_BUNDLE))
			{
				String fileName = c.getPathname();
				Properties properties = new Properties();
				
				try
				{
					fileName = ctxt.getPackageDescriptorParentUri() + File.separator + fileName;
					
					Locale locale = Locale.getDefault();
					
					if ((!locale.getLanguage().equalsIgnoreCase(DEFAULT_LANGUAGE))||(!locale.getCountry().equalsIgnoreCase(DEFAULT_COUNTRY)))
					{
						String oldFileName = fileName;
						int lastPeriod = fileName.lastIndexOf(".");
			        	if (lastPeriod==-1)
			        		fileName = fileName+"_"+locale.getLanguage()+"_"+locale.getCountry();
			        	else
			        		fileName = fileName.substring(0,lastPeriod)+"_"+locale.getLanguage()+"_"+locale.getCountry()+fileName.substring(lastPeriod,fileName.length());
			        	File f = new File(fileName);
			        	if (!f.exists())
			        		fileName = oldFileName;
					}
						
					
			        properties.load(new FileInputStream(fileName));
			        return properties;
			    }
				catch (IOException e)
				{
					System.out.println(e.getMessage());
			    }
			}
		}
		return null;
	}

	private boolean installableUnitResolution(SDDContext ctxt) {
		logReader.addLogListener(log);
		logService.log(LogService.LOG_DEBUG, "installableUnitResolution() called.");
		
		new VariableResolver(ctxt, rsrcHandler, logService).installableUnitResolver();
		RequirementsResolver reqResolver = new RequirementsResolver(ctxt, rsrcHandler);
		if(ctxt.getParameters().size() > 0) {
			if(advisor != null) {
				
				advisor.setProperties(getDescriptorLanguageBundle(ctxt));
				
				for(Object p : ctxt.getParameters().values()) {
					if(p instanceof URIParameterType) {
						ctxt.setVariable(((URIParameterType)p).getId(),
							advisor.requestParameter((URIParameterType)p).toString());
					}
					else if(p instanceof IntegerParameterType) {
						ctxt.setVariable(((IntegerParameterType)p).getId(),
							advisor.requestParameter((IntegerParameterType)p).toString());
					}
					else if(p instanceof StringParameterType) {	
						ctxt.setVariable(((StringParameterType)p).getId(),
							advisor.requestParameter((StringParameterType)p));
					}
					else if(p instanceof BooleanParameterType) {
						ctxt.setVariable(((BooleanParameterType)p).getId(),
							advisor.requestParameter((BooleanParameterType)p).toString());
					}
					
					reqResolver.installableUnitResolver();
				}
				
				if(!reqResolver.installableUnitResolver()) {
					advisor.displayError(reqResolver.getUnsatisifiedResource());
					return false;
				}
			}
			else {
				if(!reqResolver.installableUnitResolver()) {
					advisor.displayError(reqResolver.getUnsatisifiedResource());
					return false;
				}
			}
		}
		
		return true;
	}
}

