/***********************************************************************
 * Copyright (c) 2009 CA, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     CA, Inc. - Initial implementation
 *     SAS Institute - Bug 275669 
 ***********************************************************************/
package org.eclipse.cosmos.me.sdd.o10r.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.cosmos.me.sdd.schema.BooleanParameterType;
import org.eclipse.cosmos.me.sdd.schema.ContentType;
import org.eclipse.cosmos.me.sdd.schema.DeploymentDescriptorType;
import org.eclipse.cosmos.me.sdd.schema.IntegerParameterType;
import org.eclipse.cosmos.me.sdd.schema.PackageDescriptorType;
import org.eclipse.cosmos.me.sdd.schema.SDDContentPurposeType;
import org.eclipse.cosmos.me.sdd.schema.StringParameterType;
import org.eclipse.cosmos.me.sdd.schema.URIParameterType;
import org.eclipse.cosmos.me.sdd.schema.ext.ResolutionModel;
import org.eclipse.cosmos.me.sdd.schema.ext.SDDContext;
import org.eclipse.cosmos.me.sdd.schema.ext.Variable;
import org.eclipse.cosmos.me.sdd.schema.ext.Variables;
import org.eclipse.cosmos.me.sdd.schema.marshal.Marshaller;
import org.eclipse.cosmos.me.sdd.schema.marshal.MarshallerException;

public class SddContextImpl implements SDDContext {
	private static Pattern pattern = Pattern.compile( "\\$\\([a-zA-Z_]+[0-9a-zA-Z_]*\\)" );
	private static Pattern winpattern = Pattern.compile( "\\%[a-zA-Z_]+[0-9a-zA-Z_]*%" );
	private static Pattern unxpattern = Pattern.compile( "\\$\\{[a-zA-Z_]+[0-9a-zA-Z_]*\\}" );
	private Marshaller marshaller;
	private PackageDescriptorType packageDescriptor;
	private String pdParentUri;
	private DeploymentDescriptorType deploymentDescriptor;
	private Variables variables = new Variables();
	private ResolutionModel model = new ResolutionModel();
	
	public SddContextImpl(Marshaller marshaller) {
		this.marshaller = marshaller;
	}
	
	public SddContextImpl(Marshaller marshaller, String packageUri) {
		this(marshaller);
		setPackageDescriptor(packageUri);
	}
	
	public PackageDescriptorType getPackageDescriptor() {
		return packageDescriptor;
	}
	
	public void setPackageDescriptor(String packageUri) {
		packageDescriptor = loadPackageDescriptor(packageUri);
		pdParentUri = getPackageDescriptorParent(packageUri);
		loadResponseFiles();
		deploymentDescriptor = loadDeploymentDescriptor(packageDescriptor, pdParentUri);
	}
	
	public String getPackageDescriptorParentUri() {
		return pdParentUri;
	}
	
	public DeploymentDescriptorType getDeploymentDescriptor() {
		return deploymentDescriptor;
	}
	
	public String getVariable( String variable ) {
		Matcher matcher = pattern.matcher( variable );
		while ( matcher.find() ) {
			variable = variable.substring( matcher.start() + 2, matcher.end() - 1 );
		}
		
		for (Variable v : variables.getVariable()) {
			if (v.getId().equals(variable)) {
				if (v.getValue() != null) {
					return v.getValue();
				} else {
					if (v.getDefaultValue() != null) {
						return v.getDefaultValue();
					}
				}
			}
		}
		
		return "";
	}
	
	public void setVariable( String variable, String value ) {
		// if it already exists, set it
		for (Variable v : variables.getVariable()) {
			if (v.getId().equals(variable)) {
				v.setValue(value);
				return;
			}
		}
		
		// if the variable doesn't already exist, add it
		Variable v = new Variable(variable, value);
		variables.getVariable().add(v);
	}
	
	public boolean containsVariable(String variable) {
		for (Variable v : variables.getVariable()) {
			if (v.getId().equals(variable)) 
				return true;
		}
		return false;
	}

	public String substituteVariables(String expression) {
		Matcher matcher = pattern.matcher( expression );
		while ( matcher.find() ) {
			String variable = expression.substring( matcher.start() + 2, matcher.end() - 1 );
			expression = expression.replace( expression.substring( matcher.start(), matcher.end() ), getVariable( variable ) );
			matcher = pattern.matcher( expression );
		}
		
		matcher = winpattern.matcher( expression );
		while ( matcher.find() ) {
			String variable = expression.substring( matcher.start() + 1, matcher.end() - 1 );
			expression = expression.replace( expression.substring( matcher.start(), matcher.end() ), System.getenv(variable));
			matcher = pattern.matcher( expression );
		}
		
		matcher = unxpattern.matcher( expression );
		while ( matcher.find() ) {
			String variable = expression.substring( matcher.start() + 2, matcher.end() - 1 );
			expression = expression.replace( expression.substring( matcher.start(), matcher.end() ), System.getenv(variable));
			matcher = pattern.matcher( expression );
		}
		
		return expression;
	}

	public ContentType getContentById(String id) {
		for (ContentType content : packageDescriptor.getContents().getContent()) {
			if (content.getId().equals(id)) return content;
		}
		return null;
	}

	public List<ContentType> getContentByPurpose(String purpose) {
		List<ContentType> result = new ArrayList<ContentType>();
		if (packageDescriptor != null && purpose != null) {
			for (ContentType content : packageDescriptor.getContents().getContent()) {
				if (purpose.equals(content.getPurpose())) result.add(content);
			}
		}
		return result;
	}

	public InputStream getInputStreamForPath(String path) {
		// try variable substituted path
		String substitutedPath = substituteVariables(path);
		try {
			File f = new File(substitutedPath);
			if (f.canRead()) {
				return new FileInputStream(f);
			}
		} catch (FileNotFoundException e) {
			// ignore and continue
		}
		
		// prefix with package descriptor path
		String pathRelativeToPD = getPackageDescriptorParentUri() + File.separator + substitutedPath;
		try {
			File f = new File(pathRelativeToPD);
			if (f.canRead()) {
				return new FileInputStream(f);
			}
		} catch (FileNotFoundException e) {
			// ignore and continue
		}
		
		return null;
	}
	
	public ResolutionModel getResolutionModel() {
		return model;
	}
	
	public Variables getVariables() {
		return variables;
	}
	
	public void setParameter(String id, Boolean required, String defaultValue) {
		Variable v = new Variable(id, null);
		v.setParameter(true);
		if (required != null) {
			v.setRequired(required);
		}
		if (defaultValue != null) {
			v.setDefaultValue(defaultValue);
		}
		variables.getVariable().add(v);
	}

	private String getPackageDescriptorParent(String path) {
		if (path != null) {
			try {
				File dir = new File(path).getParentFile();
				if (dir.isDirectory()) {
					return dir.getCanonicalPath();
				}
			} catch (IOException e) {
				// ignore
			}
		}
		return null;
	}

	private PackageDescriptorType loadPackageDescriptor(String path) {
		if (path != null) {
			try {
				return marshaller.unmarshal(PackageDescriptorType.class, path);
			} catch (Exception e) {
				// ignore
			}
		}
		return null;
	}

	private DeploymentDescriptorType loadDeploymentDescriptor(PackageDescriptorType pd, String path) {
		if (pd != null && path != null) {
			// look for deployment descriptor
			String dd = SDDContentPurposeType.DEPLOYMENT_DESCRIPTOR.value();
			ContentType ddContent = null;
			for (ContentType ct : pd.getContents().getContent()) {
				if (dd.equals(ct.getPurpose())) {
					ddContent = ct;
					break;
				}
			}
			if (ddContent != null) {
				String ddPath = path + File.separator + ddContent.getPathname();
				return loadDeploymentDescriptor(ddPath);
			}
		}
		return null;
	}

	private DeploymentDescriptorType loadDeploymentDescriptor(String path) {
		if (path != null) {
			try {
				return marshaller.unmarshal(DeploymentDescriptorType.class, path);
			} catch (MarshallerException e) {
				// ignore
			}
		}
		return null;
	}
	
	private void loadResponseFiles() {
		List<ContentType> responseFiles = getContentByPurpose(SDDContentPurposeType.RESPONSE_FILE.value());
		for (ContentType responseFile : responseFiles) {
			InputStream stream = getInputStreamForPath(responseFile.getPathname());
			if (stream != null) {
				Properties props = new Properties();
				try {
					props.load(stream);
				} catch (IOException e) {
					e.printStackTrace();
				}
				
				for (Object key : props.keySet()) {
					String variable = (String) key;
					String value = (String) props.get(key);
					setVariable(variable, value);
				}
			}
		}
	}
	
	LinkedHashMap<String, Object> parameters  = new LinkedHashMap<String, Object>();
	
	public boolean containsParameter(String parameterId) {
		return parameters.containsKey(parameterId);
	}
	
	public void setParameter(StringParameterType type, String id,
			Boolean required, String defaultValue) {
		
		setParameter(id, required, defaultValue);		
		parameters.put(id,type);
	}

	public void setParameter(IntegerParameterType type, String id,
			Boolean required, String defaultValue) {
		setParameter(id, required, defaultValue);
		parameters.put(id,type);
	}

	public void setParameter(BooleanParameterType type, String id,
			Boolean required, String defaultValue) {
		setParameter(id, required, defaultValue);
		parameters.put(id,type);
	}

	public void setParameter(URIParameterType type, String id,
			Boolean required, String defaultValue) {
		setParameter(id, required, defaultValue);
		parameters.put(id,type);
		
	}
	
	public LinkedHashMap<String, Object> getParameters() {
		return parameters;
	}
	
	public Object getParameter(String key) {
		return parameters.get(key);
	}
}
