/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.contentunit;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Properties;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.DescribableObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.condition.ConditionImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.util.ConversionUtil;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ResultingResource;
import org.w3c.dom.Element;

public class ResultingResourceImpl extends DescribableObject implements ResultingResource {
	private static final String RESULTING_RESOURCE_TYPE = "sdd-dd:ResultingResourceType";
	private static final String RESULTING_PROPERTY_TYPE = "sdd-dd:ResultingPropertyType";
	
	public ResultingResourceImpl(Element data, SPISessionImpl session) {
		super(data, session, RESULTING_RESOURCE_TYPE);
	}

    public String getName() {
    	String name = null;
    	
    	if (isSetElement("Name")) {
    		name = getChild("Name").getTextContent();
    	}
    	
    	return name;
    }

	public String getResourceReference() {
		return getAttribute("resourceRef");
	}

	public Collection<String> getFixNames() {
		Collection<String> fixNames = new ArrayList<String>();
		
		if (isSetElement("FixName")) {
			Collection<Element> fixNameElements = getChildren("FixName");
			
			for (Element fixNameElement : fixNameElements) {
				fixNames.add(fixNameElement.getTextContent());
			}
		}

		return fixNames;
	}

	public Properties getProperties() {
    	Collection<Element> propertyElements = getChildren("Property");
    	
    	return ConversionUtil.getPropertiesFromPropertyElementList(propertyElements);
	}

	public String getVersion() {
		String version = null;
		
		if (isSetElement("Version")) {
			version = getChild("Version").getTextContent();
		}
		
		return version;
	}

	public void setName(String name) {
        if (name == null) {
            unSetElement("Name");
        }
        else {
        	Element nameElement = session.createDeploymentElement("Name");
        	nameElement.setTextContent(name);
        	
        	setContent(nameElement);
        }
    }

    public void setResourceReference(String resourceReference) {
		session.testParameter(resourceReference, 1, null);
		
		setAttribute("resourceRef", resourceReference);
	}


	public void setFixNames(Collection<String> fixNames) {
		if (fixNames == null) {
			unSetElement("FixName");
		}
		else {
			removeContent(getChildren("FixName", false));
			
			Collection<Element> fixNameElements = new ArrayList<Element>();
			
			for (String fixName : fixNames) {
				Element fixNameElement = session.createDeploymentElement("FixName");
				fixNameElement.setTextContent(fixName);
				fixNameElements.add(fixNameElement);
			}
			
			addContent(fixNameElements);
		}
	}

	public void setProperties(Properties newResultingResourceProperties) {
    	if (newResultingResourceProperties != null) {
    		Collection<Element> properties = new ArrayList<Element>();
    		removeContent(getChildren("Property", false));
    		
    		for (Enumeration<Object> e = newResultingResourceProperties.keys(); e.hasMoreElements(); ) {
    			String key = (String)e.nextElement();
    			String value = newResultingResourceProperties.getProperty(key);
    			
    			Element propertyElement = session.createDeploymentElement("Property");
    			Element propertyNameElement = session.createDeploymentElement("PropertyName");
    			Element valueElement = session.createDeploymentElement("Value");
    			
    			propertyNameElement.setTextContent(key);
    			valueElement.setTextContent(value);
                session.insertNode(propertyElement, propertyNameElement, RESULTING_PROPERTY_TYPE);
                session.insertNode(propertyElement, valueElement, RESULTING_PROPERTY_TYPE);
    			
    			properties.add(propertyElement);
    		}
    		
			addContent(properties);
    	}
    	else {
    		unSetElement("Property");
    	}
	}

	public void setVersion(String version) {
		if (version == null) {
			unSetElement("Version");
		}
		else {
			Element versionElement = session.createDeploymentElement("Version");
			versionElement.setTextContent(version);
			
			setContent(versionElement);
		}	
	}

	public String toString() {
		String resultingResourceContents = super.toString() + "Resource Reference: \"" +
				getResourceReference() + "\"\n";
		
		String version = getVersion(); 
		if (version != null) {
			resultingResourceContents += "Version: " + version + "\n";
		}
		
		String key;
		Properties resultingResourceProperties = getProperties();
		for (Enumeration<Object> e = resultingResourceProperties.keys(); e.hasMoreElements();) {
			key = (String)e.nextElement();
			resultingResourceContents += " Resulting Resource Property - Key = \"" + key + 
				"\" , value = \"" + resultingResourceProperties.getProperty(key) 
				+ "\"\n";
		}

		return resultingResourceContents;
	}
	
	
	public Object clone() {
		return new ResultingResourceImpl((Element)getElement().cloneNode(true), session);
	}


    public Condition getCondition() {
        Condition condition = null;
        
        if (isSetElement("Condition")) {
        	condition = new ConditionImpl(getChild("Condition"), session);
        }
        
        return condition;
    }


    public void setCondition(Condition condition) {
        if (condition == null) {
            unSetElement("Condition");
        }
        else {
        	setContent((SPIDataObject)condition);
        }
    }
}
