/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition;

import java.util.Collection;

 
/**
 * A VersionRange is specified with a minimum and maximum version value and a list of required fixes associated with that range. 
 * The method of comparing version strings in a version range is resource-specific.
 *
 */
public interface VersionRange {

	/**
	 * Gets the minimum version.
	 * 
	 * This is the lower bound of a version range. If MinVersion is defined but MaxVersion is not, there is no upper bound. A version that 
	 * is equal to MinVersion is within the defined range.
	 * 
	 * @return The least allowable version value.
	 */
	String getMinimumVersion();
	
	/**
	 * Gets the maximum version.
	 * 
	 * This is the upper bound of a version range. If MaxVersion is defined but MinVersion is not, there is no lower bound. A 
	 * version that is equal to MaxVersion may be within the defined range depending on the value specified for the inclusive attribute.
	 * 
	 * @return The greatest allowable version value.
	 */
	String getMaximumVersion();
	
	/**
	 * Returns whether the max version value is included in the supported range of versions.
	 * 
	 * The inclusive attribute allows the SDD author to choose the semantics of maximum version. Supported ranges are often 
	 * everything equal to or greater than the minimum version and up to, but not including, the maximum version. Sometimes it is 
	 * more convenient for the range to include the maximum version. 
	 * 
	 * @return Indicates whether the max version value is included in the supported range of versions.
	 */
	Boolean getMaximumVersionInclusiveness();
	
	/**
	 * Gets the fix names.
	 * 
	 * Any number of FixNames can be defined identifying fixes that must be found to be applied for the version constraint is to be 
	 * considered satisfied. This is true for all versions within the defined range.<br><br>
	 * When FixName is defined, either a MinVersion or a MaxVersion element MUST also be defined.
	 * 
	 * @return The collection of names of a fix.
	 */
	Collection<String> getFixNames();

	
	/**
	 * Sets the minimum version.
	 * 
	 * This is the lower bound of a version range. If MinVersion is defined but MaxVersion is not, there is no upper bound. A version that 
	 * is equal to MinVersion is within the defined range.
	 * 
	 * @param minimumVersion The least allowable version value.
	 */
	void setMinimumVersion(String minimumVersion);
	
	/**
	 * Sets the minimum version.
	 * 
	 * This is the upper bound of a version range. If MaxVersion is defined but MinVersion is not, there is no lower bound. A 
	 * version that is equal to MaxVersion may be within the defined range depending on the value specified for the inclusive attribute.
	 * 
	 * @param maximumVersion The greatest allowable version value.
	 */
	void setMaximumVersion(String maximumVersion);
	
	/**
	 * Sets whether the max version value is included in the supported range of versions.
	 * 
	 * The inclusive attribute allows the SDD author to choose the semantics of maximum version. Supported ranges are often 
	 * everything equal to or greater than the minimum version and up to, but not including, the maximum version. Sometimes it is 
	 * more convenient for the range to include the maximum version. 
	 * 
	 * @param maximumVersionInclusiveness Indicates whether the max version value is included in the supported range of versions.
	 * <ul>**default value=false</ul>
	 */
	void setMaximumVersionInclusiveness(Boolean maximumVersionInclusiveness);
	
	/**
	 * Sets the fix names.
	 * 
	 * Any number of FixNames can be defined identifying fixes that must be found to be applied for the version constraint is to be 
	 * considered satisfied. This is true for all versions within the defined range.<br><br>
	 * When FixName is defined, either a MinVersion or a MaxVersion element MUST also be defined.
	 * 
	 * @param fixNames The collection of names of a fix.
	 */
	void setFixNames(Collection<String> fixNames);
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	Object clone();
}
