/***********************************************************************
 * Copyright (c) 2009 SAS Institute, Inc. and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SAS Institute - initial API and implementation
 ***********************************************************************/
package org.eclipse.cosmos.me.sdd.profile.cim;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.text.MessageFormat;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.eclipse.cosmos.me.sdd.cr.IResourceHandler;
import org.eclipse.cosmos.me.sdd.schema.CapacityConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConditionalPropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConsumptionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.PropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.RequirementResourceConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ResourceType;
import org.eclipse.cosmos.me.sdd.schema.VersionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ext.SDDContext;

public class J2eeApplicationType implements IResourceHandler {
	protected static final QName J2EE = new QName("http://docs.oasis-open.org/sdd/ns/cim-profile", "CIM_J2eeApplication");
	protected static final QName J2EE_NAME = new QName("http://docs.oasis-open.org/sdd/ns/cim-profile", "CIM_J2eeApplication.Name");
	protected static final QName J2EE_VERSION = new QName("http://docs.oasis-open.org/sdd/ns/cim-profile", "CIM_J2eeApplication.Version");

	protected static final String CMD_DIR = "cmd.exe /c dir \"{0}\"";
	protected static final String CMD_LS = "ls -d {0}";
	protected static final String Windows = "Windows";
	protected static final Pattern PATTERN_DIR = Pattern.compile("([0-9,]+) bytes free");
	protected static final Pattern PATTERN_LS = Pattern.compile("/[\\w.*]+");
	protected static final String OS_NAME = "os.name";

	// vendor = {0}, product = {1}, version = {2}
	protected static final String WIN_1 = "{0}\\{1}\\{2}";// vendor\product\version
	protected static final String WIN_2 = "{1}\\{2}";     // product\version
	protected static final String WIN_3 = "{1}-{2}";		// product-version
	protected static final String WIN_4 = "{0}\\{1}{2}";	// vendor\productversion
	protected static final String WIN_5 = "{0}\\{1}-{2}";	// vendor\product-version
	protected static final String WIN_6 = "{1}{2}";	    // productversion


	protected static final String UNIX_1 = "{0}/{1}/{2}"; // vendor/product/version
	protected static final String UNIX_2 = "{1}/{2}";     // product/version
	protected static final String UNIX_3 = "{1}-{2}";		// product-version
	protected static final String UNIX_4 = "{0}/{1}{2}";	// vendor/productversion
	protected static final String UNIX_5 = "{0}/{1}-{2}";	// vendor/product-version
	protected static final String UNIX_6 = "{1}{2}";	    // productversion

	protected String [] unixRoots = new String [] {"/opt/", "/usr/local/"};
	protected String [] unixPatterns = new String [] {UNIX_1, UNIX_2, UNIX_3, UNIX_4, UNIX_5, UNIX_6};

	protected String [] winRoots = new String [] {"C:", "C:\\Program Files"};
	protected String [] winPatterns = new String [] {WIN_1, WIN_2, WIN_3, WIN_4, WIN_5, WIN_6};
	

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			CapacityConstraintType constraint) {
		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			ConsumptionConstraintType constraint) {
		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			PropertyConstraintType constraint) {
		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			ConditionalPropertyConstraintType constraint) {
		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			VersionConstraintType constraint) {
		return false;
	}
	
	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			RequirementResourceConstraintType constraint) {
		
		String product = null;
		String version = null;
		String vendor = null;
		
		for (Object o : constraint.getResourceConstraintGroup()) {
			if (o instanceof JAXBElement) {
				Object value = ((JAXBElement<?>)o).getValue();
		
				String constraintValue = null;
				QName name = null;
				
				if (value instanceof PropertyConstraintType) {
					if (value instanceof ConditionalPropertyConstraintType) {
						ConditionalPropertyConstraintType cc = (ConditionalPropertyConstraintType) value;
						name = cc.getPropertyName();
						constraintValue = cc.getValue();
					} else if (value instanceof PropertyConstraintType) {
						PropertyConstraintType cc = (PropertyConstraintType) value;
						name = cc.getPropertyName();
						constraintValue = cc.getValue();
					}
				}
				else if (value instanceof VersionConstraintType) {
					// not handling versions for installed product yet.
				}
				
				if (J2EE_NAME.equals(name))
					product = constraintValue;
				else if (J2EE_VERSION.equals(name))
					version = constraintValue;
			}	
		}
		
		if (System.getProperty(OS_NAME).contains(Windows)) {
			for(int i=0; i<winPatterns.length; i++) {
				String s = winFind(MessageFormat.format(winPatterns[i], vendor, product, version));
				if(s != null) {
					return true;
				}
			}
		}
		else {
			for(int i=0; i<unixPatterns.length; i++) {
				String s = unixFind(MessageFormat.format(unixPatterns[i], product, version, vendor));
				if(s != null) {
					return true;
				}
			}
		}
		
		return false;
	}
	
	public Object getPropertyValue(SDDContext ctxt, ResourceType rsrc, QName qname) {
		return null;
	}

	public Object getVersionValue(SDDContext ctxt, ResourceType rsrc) {
		return null;
	}

	protected  String unixFind(String path) {
		for(int i = 0; i < unixRoots.length; i++) {
			String s = runCommand(MessageFormat.format(CMD_LS, unixRoots[i] + path));
			
			// Make sure we got something.
			if (s != null) {
				Matcher m = PATTERN_LS.matcher(s);
				if (m.find()) {
					return unixRoots[i] + path;
				}
			}
		}
		return null;
	}

	protected  String winFind(String path) {
		for(int i = 0; i < winRoots.length; i++) {
			String s = runCommand(MessageFormat.format(CMD_DIR, winRoots[i] + "\\" + path));
			
			// Make sure we got something.
			if (s != null) {
				Matcher m = PATTERN_DIR.matcher(s);
				if (m.find()) {
					return winRoots[i] + "\\" + path;
				}
			}
		}
		return null;
	}

	private String runCommand(String command) {
		StringBuffer output = new StringBuffer();
		try 
		{ 
			// Run the command.
			Runtime runtime = Runtime.getRuntime(); 
			Process process = runtime.exec(command); 	
			if (process == null) { 
				return null;
			} 
			
			// Read the output of the command.
			BufferedReader inStream = new BufferedReader(new InputStreamReader(process.getInputStream())); 
			String line = null; 
			
			while ((line = inStream.readLine()) != null) { 
				output.append(line).append("\n");
			}  
		}
		catch (Exception e) { 
			return null; 
		} 
		
		return output.toString();
	}
}
