/***********************************************************************
 * Copyright (c) 2009 SAS Institute, Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SAS Institute, Inc. - Initial API and implementation
 ***********************************************************************/
package org.eclipse.cosmos.me.sdd.profile.cim.handler;

import java.io.File;
import java.io.FileWriter;
import java.text.MessageFormat;
import java.util.List;

import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.eclipse.cosmos.me.sdd.cr.IResourceHandler;
import org.eclipse.cosmos.me.sdd.profile.cim.DatabaseSystemType;
import org.eclipse.cosmos.me.sdd.schema.CapacityConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConditionalPropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConsumptionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.PropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.RequirementResourceConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ResourceType;
import org.eclipse.cosmos.me.sdd.schema.VersionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ext.SDDContext;

import com.sas.tools.deployjni.JNIUtils;
import com.sas.tools.deployjni.Utilities;
import com.sas.tools.deployjni.winregistry.WinRegAPI;

public class PostgresHandler extends DatabaseSystemType implements IResourceHandler {

	static final String POSTGRES = "Postgres";
	static final String POSTGRESQL = "PostgreSQL";
	static final String POSTGRES_PRODUCT_HKEY = "SOFTWARE\\PostgreSQL";
	static final String POSTGRES_VERSION_HKEY = "SOFTWARE\\PostgreSQL\\Installations\\postgresql-";
	static final String POSTGRES_WIN_LOCATION = "C:\\Program Files\\PostgreSQL";
	static final String POSTGRES_UNIX_FIND = "pgrep postgres";
	static final String POSTGRES_WIN_FIND = "pgsql-";
	static final String[] POSTGRES_PORT_CMD = { "/bin/sh", "-c", "netstat -an | grep PGSQL | grep "};
	
	static final String POSTGRES_VBS_SERVICE_CMD = "Set sh = CreateObject(\"Shell.Application\") \n"
		+ "If sh.IsServiceRunning(\"" + "SERVICE_NAME" + "\") Then \n"
		+ "   wscript.Quit(1) \n" + "End If \n"
		+ "wscript.Quit(0) \n";
	
	static final String POSTGRES_VBS_WORD_CMD= "Set sh = CreateObject(\"WScript.Shell\") \n"
		+ "wscript.Quit(sh.RegRead(\"" + "HKLM\\SOFTWARE\\PostgreSQL\\Services\\pgsql-" + "SERVICE_VERSION"
		+ "\\Port" + "\")) \n";
	
	static final String SERVICE_NAME = "SERVICE_NAME";
	static final String SERVICE_VERSION = "SERVICE_VERSION";
	static final String STOPPING = "stopping";
	
	static final String REALHOWTO ="realhowto";
	static final String VBS = ".vbs";
	static final String WSCRIPT = "wscript ";
	
	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			CapacityConstraintType constraint) {

		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			ConsumptionConstraintType constraint) {

		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			PropertyConstraintType constraint) {

		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			ConditionalPropertyConstraintType constraint) {

		return false;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			VersionConstraintType constraint) {

		return false;
	}

	public Object getPropertyValue(SDDContext ctxt, ResourceType rsrc,
			QName qname) {

		return null;
	}

	public Object getVersionValue(SDDContext ctxt, ResourceType rsrc) {
		return null;
	}

	public boolean evaluate(SDDContext ctxt, ResourceType rsrc,
			RequirementResourceConstraintType constraint) {

		List<Object> ConstraintGroup = constraint.getResourceConstraintGroup();

		for (Object o : ConstraintGroup) {
			if (o instanceof JAXBElement) {
				// go through the properties and get their values
				Object value = ((JAXBElement<?>) o).getValue();
				if (value instanceof PropertyConstraintType) {
					PropertyConstraintType prop = (PropertyConstraintType) value;

					if (prop.getPropertyName().equals(NAME)) {
						if (prop.getValue().equalsIgnoreCase(POSTGRES) == false)
							return false;
						else {
							NAME_VALUE = prop.getValue();
						}
					}

					if (prop.getPropertyName().equals(VERSION)) {
						VERSION_VALUE = prop.getValue();
					}
					if (prop.getPropertyName().equals(PORT)) {
						PORT_VALUE = prop.getValue();
					}
					if (prop.getPropertyName().equals(STATUS)) {
						STATUS_VALUE = prop.getValue();
					}
				}
			}
		}

		for (Object o : ConstraintGroup) {
			if (o instanceof JAXBElement) {
				Object value = ((JAXBElement<?>) o).getValue();
				// go through the properties once more, resolving as we walk
				if (value instanceof PropertyConstraintType) {
					PropertyConstraintType prop = (PropertyConstraintType) value;

					if (prop.getPropertyName().equals(NAME)) {
						if (resolveName() == false)
							return false;
					}
					if (prop.getPropertyName().equals(VERSION)) {
						if (resolveVersion(VERSION_VALUE) == false)
							return false;
					}
					if (prop.getPropertyName().equals(PORT)) {
						// if windows we need the version, if not we don't
						if (Utilities.getOS() == Utilities.OS_WINDOWS) {
							if (VERSION_VALUE != null) {
								if (resolvePort(VERSION_VALUE, PORT_VALUE) == false)
									return false;
							} else
								return false;
						} else if (Utilities.getOS() == Utilities.OS_LINUX) {
							if (resolvePort(null, PORT_VALUE) == false)
								return false;
						} else
							return false;
					}
					if (prop.getPropertyName().equals(STATUS)) {
						/*
						 * EXPECTED VALUES <cim:Value>OK</cim:Value>
						 * <cim:Value>Starting</cim:Value>
						 * <cim:Value>Stopping</cim:Value>
						 */
						boolean expected_status = true;

						if (STATUS_VALUE.equalsIgnoreCase(STOPPING))
							expected_status = false;

						// if windows we need the version, if not we don't
						if (Utilities.getOS() == Utilities.OS_WINDOWS) {
							if (VERSION_VALUE != null) {
								if (resolveStatus(VERSION_VALUE) != expected_status)
									return false;
							} else
								return false;
						} else if (Utilities.getOS() == Utilities.OS_LINUX) {
							if (resolveStatus(null) != expected_status)
								return false;
						} else
							return false;
					}

				}
			}
		}

		return true;
	}

	private boolean resolveName() {
		if (Utilities.getOS() == Utilities.OS_WINDOWS) {
			if (JNIUtils.libraryLoaded(getCurrentPath(), WinRegAPI.libNames)) {
				if (WinRegAPI.isWindowsRegistryKey32(
						WinRegAPI.HKEY_LOCAL_MACHINE, POSTGRES_PRODUCT_HKEY)) {
					return true;
				}
				else {
					File testDir = new File(POSTGRES_WIN_LOCATION);
					if (testDir.exists())
						return true;
				}
			}
		}

		else if (Utilities.getOS() == Utilities.OS_LINUX) {
			for (int i = 0; i < unixPatterns.length; i++) {
				String s = unixFind(MessageFormat.format(unixPatterns[i],
						POSTGRESQL, "", ""));
				if (s != null) {
					return true;
				}
			}
		}

		return false;
	}

	private boolean resolveVersion(String version) {
		if (Utilities.getOS() == Utilities.OS_WINDOWS) {
			if (JNIUtils.libraryLoaded(getCurrentPath(), WinRegAPI.libNames)) {
				if (WinRegAPI.isWindowsRegistryKey32(
						WinRegAPI.HKEY_LOCAL_MACHINE, POSTGRES_VERSION_HKEY + version)) {
					return true;
				}
				else {
					File testDir = new File(POSTGRES_WIN_LOCATION + "\\" + version);
					if (testDir.exists())
						return true;
				}
			}
		} 
		else if (Utilities.getOS() == Utilities.OS_LINUX) {
			for (int i = 0; i < unixPatterns.length; i++) {
				String s = unixFind(MessageFormat.format(unixPatterns[i],
						POSTGRESQL, version, ""));
				if (s != null) {
					return true;
				}
			}
		}

		return false;
	}

	
	private boolean isServiceRunning(String serviceName) {
		try {
			File file = File.createTempFile(REALHOWTO, VBS);
			file.deleteOnExit();
			FileWriter fw = new java.io.FileWriter(file);
		
			fw.write(getVBSServiceCmd(serviceName));
			fw.close();
			Process p = Runtime.getRuntime().exec(WSCRIPT + file.getPath());
			p.waitFor();
			return (p.exitValue() == 1);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return false;
	}
	
	private int getRegDword(String version) {

		try {
			File file = File.createTempFile(REALHOWTO, VBS);
			file.deleteOnExit();
			FileWriter fw = new java.io.FileWriter(file);
			
			fw.write(getVBSRegWordCmd(version));
			fw.close();
			Process p = Runtime.getRuntime().exec(WSCRIPT + file.getPath());
			p.waitFor();
			return (p.exitValue());
		} catch (Exception e) {
			e.printStackTrace();
		}
		return 0;
	}

	private  boolean resolveStatus(String version) {
		if (Utilities.getOS() == Utilities.OS_WINDOWS) {
			return isServiceRunning(POSTGRES_WIN_FIND + version);
		}
		else if (Utilities.getOS() == Utilities.OS_LINUX) {
			return (runCommand(POSTGRES_UNIX_FIND).length() > 0);
		} else
			return false;
	}

	private boolean resolvePort(String version, String port) {

		if (Utilities.getOS() == Utilities.OS_WINDOWS) {
			return ((Integer.toString(getRegDword(version)))
					.equalsIgnoreCase(port));
		}
		else if (Utilities.getOS() == Utilities.OS_LINUX) {
			return (runCommand(POSTGRES_PORT_CMD  + port).length() > 0);
		}
		return false;
	}
	
	private String getVBSServiceCmd(String serviceName) {
		return POSTGRES_VBS_SERVICE_CMD.replace(SERVICE_NAME, serviceName);
	}
	
	private String getVBSRegWordCmd(String version) {
		return POSTGRES_VBS_WORD_CMD.replace(SERVICE_VERSION, version);
	}
}
