/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.tests.common.spi;

import java.util.Properties;

import junit.framework.TestCase;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.SPISession;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.TranslatableString;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.OutputVariable;
import org.w3c.dom.Element;

public class OutputVariable_Test extends TestCase {
    OutputVariable var;
    OutputVariable clonedVar;
    Element element;
    SPISessionImpl session;

    public OutputVariable_Test(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
        
        var = SPISession.DEFAULT_INSTANCE.createVariableFactory().createOutputVariable("varID");
        var.setDescription(new TranslatableString("desc", "descKey"));
        var.setShortDescription(new TranslatableString("shortDesc", "shortDescKey"));
        var.setOutputParameterName("paramName");
        var.setSensitive(Boolean.TRUE);
        
        clonedVar = (OutputVariable)var.clone();
        element = ((SPIDataObject)var).getElement();
        session = (SPISessionImpl)SPISession.DEFAULT_INSTANCE;
    }

    public void testGetTranslatableStrings() {
        Properties props = var.getTranslatableStrings();
        Properties clonedProps = clonedVar.getTranslatableStrings();
        
        assertEquals(2, props.keySet().size());
        assertEquals(2, clonedProps.keySet().size());
        
        assertEquals("desc", props.get("descKey"));
        assertEquals("shortDesc", props.get("shortDescKey"));

        assertEquals("desc", clonedProps.get("descKey"));
        assertEquals("shortDesc", clonedProps.get("shortDescKey"));
    }

    public void testGetOutputParameterName() {
        assertEquals("paramName", var.getOutputParameterName());
        assertEquals("paramName", clonedVar.getOutputParameterName());
        assertEquals("paramName", element.getAttribute("outputParameterName"));
    }

    public void testGetVariableID() {
        assertEquals("varID", var.getVariableID());
        assertEquals("varID", clonedVar.getVariableID());
        assertEquals("varID", element.getAttribute("id"));
    }

    public void testIsSensitive() {
        assertTrue(var.isSensitive().booleanValue());
        assertTrue(clonedVar.isSensitive().booleanValue());
        assertTrue(Boolean.parseBoolean(element.getAttribute("sensitive")));
    }

    public void testGetDescription() {
        assertEquals("desc", var.getDescription().getValue());
        assertEquals("descKey", var.getDescription().getTranslationKey());

        assertEquals("desc", clonedVar.getDescription().getValue());
        assertEquals("descKey", clonedVar.getDescription().getTranslationKey());
        
        assertEquals("desc", (session.getChildren(element, "Description").toArray(new Element[0])[0]).getTextContent());
        assertEquals("descKey", (session.getChildren(element, "Description").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testGetShortDescription() {
        assertEquals("shortDesc", var.getShortDescription().getValue());
        assertEquals("shortDescKey", var.getShortDescription().getTranslationKey());

        assertEquals("shortDesc", clonedVar.getShortDescription().getValue());
        assertEquals("shortDescKey", clonedVar.getShortDescription().getTranslationKey());
        
        assertEquals("shortDesc", (session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getTextContent());
        assertEquals("shortDescKey", (session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testSetOutputParameterName() {
        var.setOutputParameterName("differentName");
        
        assertEquals("differentName", var.getOutputParameterName());
        assertEquals("differentName", element.getAttribute("outputParameterName"));
    }

    public void testSetSensitive() {
        var.setSensitive(Boolean.FALSE);
        
        assertFalse(var.isSensitive().booleanValue());
        assertFalse(Boolean.parseBoolean(element.getAttribute("sensitive")));
    }

    public void testSetVariableID() {
        var.setVariableID("differentID");
        
        assertEquals("differentID", var.getVariableID());
        assertEquals("differentID", element.getAttribute("id"));
        
        try {
            var.setVariableID(null);
            fail("ID was set to null");
        }
        catch (Exception e) {
        }
    }

    public void testSetDescription() {
        var.setDescription(new TranslatableString("differentDesc", "differentDescKey"));
        
        assertEquals("differentDesc", var.getDescription().getValue());
        assertEquals("differentDescKey", var.getDescription().getTranslationKey());

        assertEquals("differentDesc", (session.getChildren(element, "Description").toArray(new Element[0])[0]).getTextContent());
        assertEquals("differentDescKey", (session.getChildren(element, "Description").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testSetShortDescription()
    {
        var.setShortDescription(new TranslatableString("differentShortDesc", "differentShortDescKey"));

        assertEquals("differentShortDesc", var.getShortDescription().getValue());
        assertEquals("differentShortDescKey", var.getShortDescription().getTranslationKey());

        assertEquals("differentShortDesc", (session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getTextContent());
        assertEquals("differentShortDescKey", (session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }
}
